package com.elitescloud.boot.redis;

import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.redis.common.RedisMessageSubscribe;
import com.elitescloud.boot.redis.common.support.RedisKeyPrefix;
import com.elitescloud.boot.redis.support.RedisMessageDelegate;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.common.config.cache.CacheKeyGenerator;
import com.elitescloud.cloudt.common.constant.CacheKey;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitescloud.cloudt.common.util.RedisUtils;
import lombok.extern.log4j.Log4j2;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cache.CacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.ChannelTopic;
import org.springframework.data.redis.listener.PatternTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.util.StringUtils;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.stream.Collectors;

/**
 * Redis自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 3/14/2023
 */
@EnableConfigurationProperties({CloudtRedisProperties.class, RedisProperties.class})
@ConditionalOnClass(RedisTemplate.class)
@Log4j2
public class CloudtRedisAutoConfiguration {

    private final CloudtRedisProperties cloudtRedisProperties;

    public CloudtRedisAutoConfiguration(CloudtRedisProperties cloudtRedisProperties) {
        this.cloudtRedisProperties = cloudtRedisProperties;
    }

    @Bean
    @ConditionalOnMissingBean
    public RedisKeyPrefix redisKeyPrefixDefault() {
        return () -> CacheKey.PREFIX_DEFAULT;
    }

    @Bean
    public CacheKeyGenerator redisCacheKeyGenerator(RedisKeyPrefix redisKeyPrefix) {
        return key -> {
            String pre = "";
            if (StringUtils.hasText(cloudtRedisProperties.getPrefix())) {
                pre = cloudtRedisProperties.getPrefix() + ":";
            }
            if (redisKeyPrefix != null) {
                String keyPrefix = redisKeyPrefix.getKeyPrefix();
                if (StringUtils.hasText(keyPrefix)) {
                    pre = pre + keyPrefix + ":";
                }
            }

            if (key.indexOf(pre) == 0) {
                return key;
            }
            return pre + key;
        };
    }

    @Bean
    @Primary
    public RedisTemplate redisTemplate(RedisConnectionFactory redisConnectionFactory, CacheKeyGenerator cacheKeyGenerator) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();

        RedisSerializer<String> redisSerializer = redisSerializer(cacheKeyGenerator);

        template.setConnectionFactory(redisConnectionFactory);
        template.setKeySerializer(redisSerializer);
        template.setHashKeySerializer(new StringRedisSerializer());
        template.setValueSerializer(new JdkSerializationRedisSerializer());
        template.setHashValueSerializer(new JdkSerializationRedisSerializer());
        return template;
    }


    @Bean
    public RedisUtils redisUtils(RedisTemplate<Object, Object> redisTemplate,
                                 CacheKeyGenerator keyGenerator) {
        return new RedisUtils(redisTemplate, keyGenerator);
    }

    @Bean
    public RedLockUtils redLockUtils(CacheKeyGenerator keyGenerator,
                                     ObjectProvider<RedissonClient> redissonClientObjectProvider) {
        return new RedLockUtils(keyGenerator, redissonClientObjectProvider);
    }

    @Bean
    @ConditionalOnMissingBean(RedisWrapper.class)
    public RedisWrapper redisWrapper() {
        return (supplier, param) -> supplier.get();
    }

    @Bean
    @Primary
    public CacheManager redisCacheManager(RedisConnectionFactory redisConnectionFactory, CacheKeyGenerator cacheKeyGenerator) {
        return new RedisCacheManager(RedisCacheWriter.lockingRedisCacheWriter(redisConnectionFactory),
                RedisCacheConfiguration.defaultCacheConfig()
                        .disableCachingNullValues()
                        .computePrefixWith(cacheKeyGenerator::computeKey)
        );
    }

    @Bean
    @ConditionalOnBean(RedisMessageSubscribe.class)
    public RedisMessageListenerContainer redisMessageListenerContainer(RedisConnectionFactory redisConnectionFactory,
                                                                       RedisTemplate redisTemplate,
                                                                       ObjectProvider<RedisMessageSubscribe> subscribeObjectProvider) {
        var container = new RedisMessageListenerContainer();
        container.setConnectionFactory(redisConnectionFactory);

        for (RedisMessageSubscribe redisMessageSubscribe : subscribeObjectProvider) {
            this.addSubscriber(container, redisMessageSubscribe, redisTemplate.getStringSerializer(), redisTemplate.getValueSerializer());
        }
        
        return container;
    }

    private void addSubscriber(RedisMessageListenerContainer container, RedisMessageSubscribe subscribe,
                               RedisSerializer<String> channelSerializer, RedisSerializer<?> valueSerializer) {
        if (ArrayUtil.isEmpty(subscribe.channel()) && ArrayUtil.isEmpty(subscribe.channelPattern())) {
            throw new IllegalStateException(subscribe.getClass().getName() + "未配置监听的channel");
        }
        if (ArrayUtil.isNotEmpty(subscribe.channel())) {
            var topics = Arrays.stream(subscribe.channel()).distinct().map(ChannelTopic::new).collect(Collectors.toList());
            container.addMessageListener(new RedisMessageDelegate(subscribe, channelSerializer, valueSerializer), topics);
        }
        if (ArrayUtil.isNotEmpty(subscribe.channelPattern())) {
            var topics = Arrays.stream(subscribe.channelPattern()).distinct().map(PatternTopic::new).collect(Collectors.toList());
            container.addMessageListener(new RedisMessageDelegate(subscribe, channelSerializer, valueSerializer), topics);
        }
    }

    private RedisSerializer<String> redisSerializer(CacheKeyGenerator cacheKeyGenerator) {
        return new RedisSerializer<>() {
            private final Charset charset = StandardCharsets.UTF_8;

            @Override
            public String deserialize(byte[] bytes) {
                if (bytes == null) {
                    return null;
                }

                return new String(bytes, charset);
            }

            @Override
            public byte[] serialize(String key) {
                if (key == null) {
                    return new byte[0];
                }

                key = cacheKeyGenerator.computeKey(key);
                return key.getBytes(charset);
            }
        };
    }
}
