package com.elitescloud.boot.redis;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.redis.redisson.RedissonCustomProperties;
import lombok.extern.log4j.Log4j2;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.spring.cache.CacheConfig;
import org.redisson.spring.cache.RedissonSpringCacheManager;
import org.redisson.spring.data.connection.RedissonConnectionFactory;
import org.redisson.spring.starter.RedissonAutoConfiguration;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.cache.CacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/23
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnClass(Redisson.class)
@ConditionalOnProperty(prefix = RedissonCustomProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@AutoConfigureBefore({RedissonAutoConfiguration.class, CloudtRedisAutoConfiguration.class})
@Log4j2
class CloudtRedissonAutoConfiguration {

    private static final String REDIS_PROTOCOL_PREFIX = "redis://";
    private static final String REDISS_PROTOCOL_PREFIX = "rediss://";

    private final RedissonCustomProperties redissonCustomProperties;
    private final RedisProperties redisProperties;

    public CloudtRedissonAutoConfiguration(CloudtRedisProperties properties, RedisProperties redisProperties) {
        this.redissonCustomProperties = properties.getRedisson();
        this.redisProperties = redisProperties;
    }

    @Bean(destroyMethod = "shutdown")
    @Primary
    public RedissonClient redissonClient() {
        // 优先从redisson配置中取
        Config config = redissonCustomProperties.buildConfig();
        if (config == null) {
            // 其次从redis配置中取
            config = buildRedissonConfigByRedis();
        }

        return Redisson.create(config);
    }

    @Bean
    public RedisConnectionFactory redisConnectionFactory(RedissonClient redissonClient) {
        return new RedissonConnectionFactory(redissonClient);
    }

    @Bean
    public CacheManager redissonManager(RedissonClient redissonClient, CloudtRedisProperties cloudtRedisProperties) {
        Map<String, CacheConfig> configHashedMap = new HashMap<>();

        String prefix = "";
        if (StringUtils.hasText(cloudtRedisProperties.getPrefix())) {
            prefix = cloudtRedisProperties.getPrefix() + ":";
        }

        configHashedMap.put(prefix + "longCache", new CacheConfig(180000, 120000));
        configHashedMap.put(prefix + "shortCache", new CacheConfig(60000, 30000));
        return new RedissonSpringCacheManager(redissonClient, configHashedMap);
    }

    private Config buildRedissonConfigByRedis() {
        int timeout = (int) obtainTimeout();

        Config config = null;
        if (redisProperties.getSentinel() != null) {
            String[] nodes = convert(redisProperties.getSentinel().getNodes());
            config = new Config();
            config.useSentinelServers()
                    .setMasterName(redisProperties.getSentinel().getMaster())
                    .addSentinelAddress(nodes)
                    .setDatabase(redisProperties.getDatabase())
                    .setConnectTimeout(timeout)
                    .setPassword(CharSequenceUtil.blankToDefault(redisProperties.getPassword(), null));
        } else if (redisProperties.getCluster() != null) {
            String[] nodes = convert(redisProperties.getCluster().getNodes());

            config = new Config();
            config.useClusterServers()
                    .addNodeAddress(nodes)
                    .setConnectTimeout(timeout)
                    .setPassword(CharSequenceUtil.blankToDefault(redisProperties.getPassword(), null));
        } else {
            config = new Config();
            String prefix = REDIS_PROTOCOL_PREFIX;
            if (redisProperties.isSsl()) {
                prefix = REDISS_PROTOCOL_PREFIX;
            }

            config.useSingleServer()
                    .setAddress(prefix + redisProperties.getHost() + ":" + redisProperties.getPort())
                    .setConnectTimeout(timeout)
                    .setDatabase(redisProperties.getDatabase())
                    .setPassword(CharSequenceUtil.blankToDefault(redisProperties.getPassword(), null))
                    .setConnectionMinimumIdleSize(6)
            ;
        }
        return config;
    }

    private long obtainTimeout() {
        Duration timeoutValue = redisProperties.getTimeout();
        return timeoutValue == null ? 10000 : timeoutValue.toMillis();
    }

    private String[] convert(List<String> nodesObject) {
        List<String> nodes = new ArrayList<>(nodesObject.size());
        for (String node : nodesObject) {
            if (!node.startsWith(REDIS_PROTOCOL_PREFIX) && !node.startsWith(REDISS_PROTOCOL_PREFIX)) {
                nodes.add(REDIS_PROTOCOL_PREFIX + node);
            } else {
                nodes.add(node);
            }
        }
        return nodes.toArray(new String[0]);
    }
}
