package com.elitescloud.boot.security.config.metadata;

import com.elitescloud.boot.security.common.support.PermissionMetadataProvider;
import com.elitescloud.boot.provider.CurrentUserProvider;
import com.elitescloud.boot.security.config.CustomSecurityProperties;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.provider.dto.SysApiPermissionMetadataDTO;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpMethod;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.SecurityConfig;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.servlet.util.matcher.MvcRequestMatcher;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.handler.HandlerMappingIntrospector;

import javax.servlet.http.HttpServletRequest;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/7
 */
@Log4j2
public class CloudtFilterInvocationSecurityMetadataSource implements FilterInvocationSecurityMetadataSource {
    static final String DENIED = "DENIED";

    private final CustomSecurityProperties securityProperties;
    private final CurrentUserProvider currentUserProvider;
    private final PermissionMetadataProvider permissionMetadataProvider;

    /**
     * 是否白名单
     */
    private Predicate<HttpServletRequest> allowPredicate = t -> false;
    /**
     * 是否可匿名访问
     */
    private boolean anonymous = true;
    private HandlerMappingIntrospector handlerMappingIntrospector = new HandlerMappingIntrospector();

    /**
     * 已认证过的请求的缓存
     */
    private final Cache<String, Collection<ConfigAttribute>> requestPermissionCache;
    /**
     * 租户下的权限
     */
    private final Cache<String, List<SysApiPermissionMetadataDTO>> tenantPermissionRoleCache;

    public CloudtFilterInvocationSecurityMetadataSource(CustomSecurityProperties securityProperties,
                                                        CurrentUserProvider currentUserProvider,
                                                        PermissionMetadataProvider permissionMetadataProvider) {
        this.securityProperties = securityProperties;
        this.currentUserProvider = currentUserProvider;
        this.permissionMetadataProvider = permissionMetadataProvider;

        requestPermissionCache = Caffeine.newBuilder()
                .maximumSize(500)
                .expireAfterWrite(securityProperties.getPermissionActionCache())
                .build();
        tenantPermissionRoleCache = Caffeine.newBuilder()
                .maximumSize(100)
                .expireAfterWrite(securityProperties.getPermissionActionCache())
                .build();
    }

    @Override
    public Collection<ConfigAttribute> getAttributes(Object object) throws IllegalArgumentException {
        // 如果没有开启权限校验，则直接返回空，返回空会直接跳过之后的角色校验
        if (!Boolean.TRUE.equals(securityProperties.getPermissionEnabled())) {
            return SecurityConfig.createList();
        }
        HttpServletRequest request = ((FilterInvocation) object).getRequest();

        // 判断是否在白名单
        if (allowPredicate != null && allowPredicate.test(request)) {
            return SecurityConfig.createList();
        }

        // 是否匿名访问
        var currentUser = currentUserProvider.currentUser();
        if (currentUser == null && !anonymous) {
            // 不允许匿名访问
            return decline();
        }
        if (currentUser != null && currentUser.isSystemAdmin()) {
            // 系统管理员不受限
            return SecurityConfig.createList();
        }

        // 先从本地缓存查询
        var cacheKey = generateCacheKey(currentUser) + ":" + request.getMethod() + ":" + request.getRequestURI();
        var configAttributes = requestPermissionCache.getIfPresent(cacheKey);
        if (configAttributes == null) {
            // 本地缓存没有，则查询
            configAttributes = queryAttribute(currentUser, request);
            // 加入本地缓存
            requestPermissionCache.put(cacheKey, configAttributes);
        }

        return SecurityConfig.createList();
    }

    @Override
    public Collection<ConfigAttribute> getAllConfigAttributes() {
        return null;
    }

    @Override
    public boolean supports(Class<?> clazz) {
        return FilterInvocation.class.isAssignableFrom(clazz);
    }

    public void setAllowPredicate(Predicate<HttpServletRequest> allowPredicate) {
        this.allowPredicate = allowPredicate;
    }

    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    public void setHandlerMappingIntrospector(HandlerMappingIntrospector handlerMappingIntrospector) {
        this.handlerMappingIntrospector = handlerMappingIntrospector;
    }

    private Collection<ConfigAttribute> decline() {
        return SecurityConfig.createList((securityProperties.getRolePrefix() + DENIED));
    }

    private Collection<ConfigAttribute> queryAttribute(GeneralUserDetails currentUser, HttpServletRequest request) {
        var permissions = queryPermissionMetadata(currentUser);
        if (permissions.isEmpty()) {
            return Collections.emptyList();
        }

        return permissions.stream()
                .filter(t -> {
                    if (!StringUtils.hasText(t.getUri())) {
                        return false;
                    }

                    if (securityProperties.isMvcMatcher()) {
                        var matcher = new MvcRequestMatcher(handlerMappingIntrospector, t.getUri());
                        if (StringUtils.hasText(t.getMethod())) {
                            matcher.setMethod(HttpMethod.resolve(t.getMethod().toUpperCase()));
                        }
                        return matcher.matches(request);
                    }

                    // ant匹配
                    var matcher = new AntPathRequestMatcher(t.getUri(), StringUtils.hasText(t.getMethod()) ? t.getMethod().toUpperCase() : null);
                    return matcher.matches(request);
                })
                .flatMap(t -> {
                    if (CollectionUtils.isEmpty(t.getRoleCodes())) {
                        // 未配置角色，直接拒绝
                        return Stream.of(securityProperties.getRolePrefix() + DENIED);
                    }
                    return t.getRoleCodes().stream().map(tt -> securityProperties.getRolePrefix() + tt);
                }).distinct()
                .map(SecurityConfig::new)
                .collect(Collectors.toList());
    }

    private List<SysApiPermissionMetadataDTO> queryPermissionMetadata(GeneralUserDetails currentUser) {
        var cacheKey = generateCacheKey(currentUser);

        // 先从本地缓存查询
        var permissions = tenantPermissionRoleCache.getIfPresent(cacheKey);
        if (permissions != null) {
            return permissions;
        }

        // 远程查询
        permissions = permissionMetadataProvider.queryPermissionMetadata(currentUser);
        if (permissions == null) {
            permissions = Collections.emptyList();
        }
        tenantPermissionRoleCache.put(cacheKey, permissions);

        return permissions;
    }

    private String generateCacheKey(GeneralUserDetails currentUser) {
        String key = "default";
        if (currentUser != null && currentUser.getTenant() != null) {
            key = currentUser.getTenant().getTenantCode();
        }

        return key;
    }
}
