package com.elitescloud.boot.security.config;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.client.common.InterceptUri;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.security.common.support.PermissionMetadataProvider;
import com.elitescloud.boot.security.config.metadata.CloudtFilterInvocationSecurityMetadataSource;
import com.elitescloud.boot.security.config.metadata.CloudtRoleVoter;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.security.access.AccessDecisionManager;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.AffirmativeBased;
import org.springframework.security.access.vote.RoleVoter;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.servlet.util.matcher.MvcRequestMatcher;
import org.springframework.web.servlet.handler.HandlerMappingIntrospector;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

/**
 * @author Michael Li
 */
@Log4j2
public class FilterInvocationSecurityConfig {

    private final AuthorizationProperties authorizationProperties;
    private final CustomSecurityProperties customSecurityProperties;

    public FilterInvocationSecurityConfig(AuthorizationProperties authorizationProperties, CustomSecurityProperties customSecurityProperties) {
        this.authorizationProperties = authorizationProperties;
        this.customSecurityProperties = customSecurityProperties;
    }

    @Bean
    public RoleVoter roleVoter() {
        return new CloudtRoleVoter(customSecurityProperties);
    }

    @Bean
    public AccessDecisionManager accessDecisionManager(List<AccessDecisionVoter<?>> decisionVoters) {
        return new AffirmativeBased(decisionVoters);
    }

//    @Bean
//    public FilterInvocationSecurityMetadataSource filterInvocationSecurityMetadataSource(CustomSecurityProperties customSecurityProperties,
//                                                                                         CurrentUserProvider currentUserProvider) {
//        // 设置接口控制的 MetadataSource 类
//        return new PermissionBasedFilterSecurityMetadataSource(customSecurityProperties, currentUserProvider);
//    }

    @Bean
    public FilterInvocationSecurityMetadataSource filterInvocationSecurityMetadataSource(CustomSecurityProperties customSecurityProperties,
                                                                                         UserDetailProvider currentUserProvider,
                                                                                         @Autowired(required = false) RedisUtils redisUtils,
                                                                                         PermissionMetadataProvider permissionMetadataProvider,
                                                                                         HandlerMappingIntrospector handlerMappingIntrospector) {
        // 设置接口控制的 MetadataSource 类
        var metadataSource = new CloudtFilterInvocationSecurityMetadataSource(customSecurityProperties, currentUserProvider
                , permissionMetadataProvider);
        metadataSource.setAnonymous(ObjectUtil.defaultIfNull(authorizationProperties.getAnonymousEnabled(), false));
        metadataSource.setAllowPredicate(allowPredicate(handlerMappingIntrospector));
        metadataSource.setHandlerMappingIntrospector(handlerMappingIntrospector);
        return metadataSource;
    }

    @Bean
    @ConditionalOnMissingBean
    public PermissionMetadataProvider permissionMetadataProvider() {
        log.warn("未发现有效的PermissionMetadataProvider");
        return currentUser -> Collections.emptyList();
    }

    private Predicate<HttpServletRequest> allowPredicate(HandlerMappingIntrospector handlerMappingIntrospector) {
        // 白名单
        Set<String> allowUris = new HashSet<>(64);
        allowUris.addAll(InterceptUri.getAllowUri());
        allowUris.addAll(ObjectUtil.defaultIfNull(authorizationProperties.getAllowList(), Collections.emptySet()));
        if (allowUris.isEmpty()) {
            return t -> false;
        }

        return request -> allowUris.stream().anyMatch(t -> new MvcRequestMatcher(handlerMappingIntrospector, t).matches(request));
    }
}
