package org.apache.shardingsphere.sql.parser.binder.segment.select.pagination;

import lombok.Generated;
import lombok.extern.slf4j.Slf4j;
import org.apache.shardingsphere.sql.parser.binder.statement.dml.SelectStatementContext;
import org.apache.shardingsphere.sql.parser.sql.segment.dml.pagination.NumberLiteralPaginationValueSegment;
import org.apache.shardingsphere.sql.parser.sql.segment.dml.pagination.PaginationValueSegment;
import org.apache.shardingsphere.sql.parser.sql.segment.dml.pagination.ParameterMarkerPaginationValueSegment;
import org.apache.shardingsphere.sql.parser.sql.segment.dml.pagination.limit.LimitValueSegment;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 解决分页时actualOffset处理{@link #getValue(PaginationValueSegment, List)}的bug.
 * <p>
 * 场景：使用querydsl分页查询时offset的参数位置解析错误，如以下SQL (条件中price字段是BigDecimal类型)：
 * <pre>
 *      QTestObjDO objDO = QTestObjDO.testObjDO;
 *      var condition = objDO.price.subtract(0).goe(0);
 *      var query = jpaQueryFactory.select(objDO)
 *                 .from(objDO)
 *                 .where(condition)
 *                 .offset(1).limit(3);
 *      var list = query.fetch();
 *</pre>
 *</p>
 * @author Kaiser（wang shao）
 * @date 2022/2/17
 */
@Slf4j
public final class PaginationContext {
    private final boolean hasPagination;
    private final PaginationValueSegment offsetSegment;
    private final PaginationValueSegment rowCountSegment;
    private final long actualOffset;
    private final Long actualRowCount;

    public PaginationContext(PaginationValueSegment offsetSegment, PaginationValueSegment rowCountSegment, List<Object> parameters) {
        this.hasPagination = null != offsetSegment || null != rowCountSegment;
        this.offsetSegment = offsetSegment;
        this.rowCountSegment = rowCountSegment;
        this.actualOffset = null == offsetSegment ? 0L : this.getValue(offsetSegment, parameters);
        this.actualRowCount = null == rowCountSegment ? null : this.getValue(rowCountSegment, parameters);
    }

    private long getValue(PaginationValueSegment paginationValueSegment, List<Object> parameters) {
        if (paginationValueSegment instanceof ParameterMarkerPaginationValueSegment) {
            Object obj = parameters.get(((ParameterMarkerPaginationValueSegment) paginationValueSegment).getParameterIndex());
            //fix
//            return obj instanceof Long ? (Long) obj : (long) (Integer) obj;
            if (obj instanceof Long) {
                return (long) obj;
            }
            try {
                return Long.parseLong(Objects.requireNonNullElse(obj, "0").toString());
            } catch (NumberFormatException e) {
                log.error("分页参数处理bug，可能是sharding本身的bug，错误信息：", e);
                return 0;
            }
        } else {
            return ((NumberLiteralPaginationValueSegment) paginationValueSegment).getValue();
        }
    }

    public Optional<PaginationValueSegment> getOffsetSegment() {
        return Optional.ofNullable(this.offsetSegment);
    }

    public Optional<PaginationValueSegment> getRowCountSegment() {
        return Optional.ofNullable(this.rowCountSegment);
    }

    public long getActualOffset() {
        if (null == this.offsetSegment) {
            return 0L;
        } else {
            return this.offsetSegment.isBoundOpened() ? this.actualOffset - 1L : this.actualOffset;
        }
    }

    public Optional<Long> getActualRowCount() {
        return null == this.rowCountSegment ? Optional.empty() : Optional.of(this.rowCountSegment.isBoundOpened() ? this.actualRowCount + 1L : this.actualRowCount);
    }

    public Optional<Integer> getOffsetParameterIndex() {
        return this.offsetSegment instanceof ParameterMarkerPaginationValueSegment ? Optional.of(((ParameterMarkerPaginationValueSegment) this.offsetSegment).getParameterIndex()) : Optional.empty();
    }

    public Optional<Integer> getRowCountParameterIndex() {
        return this.rowCountSegment instanceof ParameterMarkerPaginationValueSegment ? Optional.of(((ParameterMarkerPaginationValueSegment) this.rowCountSegment).getParameterIndex()) : Optional.empty();
    }

    public long getRevisedOffset() {
        return 0L;
    }

    public long getRevisedRowCount(SelectStatementContext shardingStatement) {
        if (this.isMaxRowCount(shardingStatement)) {
            return 2147483647L;
        } else {
            return this.rowCountSegment instanceof LimitValueSegment ? this.actualOffset + this.actualRowCount : this.actualRowCount;
        }
    }

    private boolean isMaxRowCount(SelectStatementContext shardingStatement) {
        return (!shardingStatement.getGroupByContext().getItems().isEmpty() || !shardingStatement.getProjectionsContext().getAggregationProjections().isEmpty()) && !shardingStatement.isSameGroupByAndOrderByItems();
    }

    @Generated
    public boolean isHasPagination() {
        return this.hasPagination;
    }
}
