package com.elitescloud.boot.swagger.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.net.NetUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.ClassNameConstant;
import com.elitescloud.boot.constant.WebConstant;
import com.elitescloud.boot.util.CloudtBootUtil;
import com.github.xiaoymin.knife4j.spring.extension.OpenApiExtensionResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpHeaders;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import springfox.bean.validators.configuration.BeanValidatorPluginsConfiguration;
import springfox.documentation.RequestHandler;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.ParameterBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.schema.ModelRef;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Contact;
import springfox.documentation.service.Parameter;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.annotations.EnableSwagger2WebMvc;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;

/**
 * swagger的基础配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/01
 */
@EnableSwagger2WebMvc
@Import(BeanValidatorPluginsConfiguration.class)
public abstract class BaseSwaggerConfig {
    private static final Logger log = LoggerFactory.getLogger(BaseSwaggerConfig.class);

    /**
     * 服务地址
     */
    @Value("${swagger.host:#{''}}")
    protected String host;

    /**
     * 服务地址
     */
    @Value("${swagger.base-path:#{'/'}}")
    protected String basePath;

    /**
     * 标题
     */
    @Value("${swagger.title:#{''}}")
    protected String title;
    /**
     * 描述
     */
    @Value("${swagger.description:#{''}}")
    protected String description;
    /**
     * URL
     */
    @Value("${swagger.url:#{''}}")
    protected String url;
    /**
     * 作者
     */
    @Value("${swagger.contact.name:#{''}}")
    protected String contactName;
    /**
     * 作者网址
     */
    @Value("${swagger.contact.url:#{''}}")
    protected String contactUrl;
    /**
     * 作者邮箱
     */
    @Value("${swagger.contact.email:#{''}}")
    protected String contactEmail;

    @Value("${swagger.exclude.rpc:#{true}}")
    protected Boolean excludeRpc;

    protected final OpenApiExtensionResolver openApiExtensionResolver;
    protected final ServerProperties serverProperties;

    protected BaseSwaggerConfig(OpenApiExtensionResolver openApiExtensionResolver, ServerProperties serverProperties) {
        this.openApiExtensionResolver = openApiExtensionResolver;
        this.serverProperties = serverProperties;
    }

    protected Docket createDocket(@NonNull String groupName, @NonNull String... controllerPackage) {
        return createDocket(groupName, null, controllerPackage);
    }

    protected Docket createDocket(@NonNull String groupName, List<Parameter> globalParameters, @NonNull String... controllerPackage) {
        Assert.hasText(groupName, "swagger配置失败，groupName为空");
        Assert.notEmpty(controllerPackage, "swagger配置失败，缺少controller包路径");

        log.info("create Docket:{}, {}", groupName, String.join(",", controllerPackage));
        // 公共参数
        List<Parameter> parameters = new ArrayList<>();

        // token
        parameters.add(parameterGlobal(HttpHeaders.AUTHORIZATION, 0, "用户token"));
        if (CollUtil.isNotEmpty(globalParameters)) {
            parameters.addAll(globalParameters);
        }

        parameters.add(parameterGlobal(WebConstant.HEADER_MENU_CODE, 1, "菜单编码"));
        parameters.add(parameterGlobal(WebConstant.HEADER_BUSINESS_OBJECT, 2, "业务对象编码"));
        parameters.add(parameterGlobal(WebConstant.HEADER_BUSINESS_OPERATION, 3, "业务对象的操作编码"));

        // 租户ID
        try {
            Class.forName(ClassNameConstant.TENANT_CLIENT);
            var paramTenantId = new ParameterBuilder().name(WebConstant.HEADER_TENANT_CODE)
                    .parameterType("header")
                    .modelRef(new ModelRef("string"))
                    .order(1)
                    .description("租户编码")
                    .build();
            parameters.add(paramTenantId);
        } catch (Exception ignored) {
        }

        // api路径
        Predicate<RequestHandler> predicate = null;
        for (String p : controllerPackage) {
            if (predicate == null) {
                predicate = RequestHandlerSelectors.basePackage(p);
                continue;
            }

            predicate = predicate.or(RequestHandlerSelectors.basePackage(p));
        }

        return new Docket(DocumentationType.SWAGGER_2)
                .groupName(groupName)
                .apiInfo(apiInfo(groupName))
                .select()
                .apis(predicate)
                .paths(pathSelector())
                .build()
                .host(host)
                .pathMapping(basePath)
                .globalOperationParameters(parameters)
                .extensions(openApiExtensionResolver.buildExtensions(groupName))
                ;
    }

    /**
     * 路径选择器
     *
     * @return
     */
    protected Predicate<String> pathSelector() {
        // 去除掉rpc的
        if (excludeRpc()) {
            return p -> !p.startsWith("/rpc");
        }

        return PathSelectors.any();
    }

    /**
     * 是否排除掉RPC
     *
     * @return
     */
    protected boolean excludeRpc() {
        return excludeRpc == null || excludeRpc;
    }

    protected ApiInfo apiInfo(String desc) {
        return new ApiInfoBuilder()
                .title(title)
                .description(description + "【" + desc + "】")
                .termsOfServiceUrl(getServiceUrl())
                .contact(new Contact(contactName, contactUrl, contactEmail))
                .version(CloudtBootUtil.getProjectVersion())
                .build();
    }

    protected Parameter parameterGlobal(String name, Integer order, String description) {
        return new ParameterBuilder().name(name)
                .parameterType("header")
                .modelRef(new ModelRef("String"))
                .order(order)
                .description(description)
                .build();
    }

    protected String getServiceUrl() {
        if (CharSequenceUtil.isNotBlank(url)) {
            return url;
        }
        int port = Objects.requireNonNullElse(serverProperties.getPort(), 8080);
        var contextPath = CharSequenceUtil.blankToDefault(serverProperties.getServlet().getContextPath(), "");
        return "http://" + getIp() + ":" + port + contextPath;
    }

    protected String getIp() {
        if (serverProperties.getAddress() != null) {
            return serverProperties.getAddress().getHostAddress();
        }

        return ObjectUtil.defaultIfNull(NetUtil.getLocalhost().getHostAddress(), "127.0.0.1");
    }
}
