package com.elitescloud.boot.swagger.openapi.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.NamingCase;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.BusinessParam;
import com.elitescloud.boot.common.annotation.businessobject.OperationScopeEnum;
import com.elitescloud.boot.common.annotation.businessobject.OperationStateEnum;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.constant.WebConstant;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.swagger.openapi.common.spi.OpenApiLoadStrategySPI;
import com.elitescloud.boot.swagger.openapi.config.OpenApiProperties;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectInfo;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectOperationInfo;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectParamInfo;
import com.elitescloud.boot.swagger.openapi.swagger3.annotations.enums.ParameterIn;
import com.elitescloud.boot.swagger.openapi.swagger3.core.converter.ModelConverters;
import com.elitescloud.boot.swagger.openapi.swagger3.core.converter.ResolvedSchema;
import com.elitescloud.boot.swagger.openapi.swagger3.core.util.PrimitiveType;
import com.elitescloud.boot.swagger.openapi.swagger3.models.*;
import com.elitescloud.boot.swagger.openapi.swagger3.models.info.Info;
import com.elitescloud.boot.swagger.openapi.swagger3.models.media.*;
import com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.HeaderParameter;
import com.elitescloud.boot.swagger.openapi.swagger3.models.responses.ApiResponse;
import com.elitescloud.boot.swagger.openapi.swagger3.models.responses.ApiResponses;
import com.elitescloud.boot.swagger.openapi.swagger3.models.tags.Tag;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.cloudt.common.annotation.Comment;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.persistence.*;
import javax.validation.constraints.NotNull;
import java.lang.reflect.Parameter;
import java.lang.reflect.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description : openApi规范的对象封装，结合业务对象扩展
 * @date : 2024-01-02 15:00
 */
@Slf4j
public class OpenApiServiceUtil {
    public static final String X_PROJECT = "x-Project";
    public static final String X_PROJECT_NAME = "x-Project-Name";
    public static final String X_ENV = "x-Env";
    public static final String X_ENV_NAME = "x-Env-Name";
    public static final String X_SERVICE = "x-Service";
    public static final String X_SERVICE_NAME = "x-Service-Name";
    public static final String X_BUSINESS_OBJECTS = "x-BusinessObjects";

    public static final String X_BUSINESS_OBJECT = "x-BusinessObject";
    public static final String X_BUSINESS_OBJECT_OPERATION = "x-BusinessObject-Operation";
    public static final String X_OPEN_API_PROPERTIES = "x-OpenApiProperties";
    public static final String X_PROJECT_VERSION = "x-ProjectVersion";
    public static final String X_CLOUDT_BOOT_VERSION = "x-CloudtBootVersion";
    public static final String X_NEW_DATE = "x-BuildTime";
    public static final String DEFAULT_TAG = "公共接口";

    private static final List<com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter> commonHeaders = new ArrayList<>();
    private static final Map<Class<?>, BusinessObjectParamInfo> PARAM_INFO_MAP = new HashMap<>(128);

    private static final List<OpenApiLoadStrategySPI> spiList = new ArrayList<>(4);
    private static final ModelConverters modelConverters = ModelConverters.getInstance();

    static {
        commonHeaders.add(new HeaderParameter().name(HttpHeaders.AUTHORIZATION).required(true).description("认证令牌").schema(new StringSchema()));
        commonHeaders.add(new HeaderParameter().name(TenantConstant.HEADER_TENANT_CODE).required(false).description("租户编码:操作者如果是第三方客户端（或者其它未登录用户的情况下），需要指定访问某租户下的数据时传递").schema(new StringSchema()));
        commonHeaders.add(new HeaderParameter().name(TenantConstant.HEADER_TENANT_ID).required(false).description("同理X-Tenant-Code，目前在用的业务场景：工作流回调业务时传递的租户ID").schema(new StringSchema()));
        commonHeaders.add(new HeaderParameter().name(WebConstant.HEADER_MENU_CODE).required(false).description("菜单编码：用于数据权限，传递请求API是哪个菜单").schema(new StringSchema()));
        commonHeaders.add(new HeaderParameter().name(WebConstant.HEADER_BUSINESS_OBJECT).required(false).description("业务对象编码：用于数据权限传递属于哪个业务对象").schema(new StringSchema()));
        commonHeaders.add(new HeaderParameter().name(WebConstant.HEADER_BUSINESS_OPERATION).required(false).description("业务对象的操作编码：用于数据权限传递属于业务对象的哪个操作").schema(new StringSchema()));
    }


    //设置操作header头信息内容
    public static void processOperation(Operation operation) {
        if (operation != null) {
            // 通用请求头
            commonHeaders.forEach(operation::addParametersItem);
        }
    }

    //设置Openapi信息
    public static void setInitInfo(OpenAPI openAPI, OpenApiProperties openApiProperties) {
        Paths paths = new Paths();
        Tag tag = new Tag()
                .name(DEFAULT_TAG)
                .description("未业务对象注解进行分组的接口，公共接口没有具体业务对象。");
        openAPI.addTagsItem(tag);
        for (OperationStateEnum state : OperationStateEnum.values()) {
            openAPI.addTagsItem(new Tag().name(state.getDescription()).description(state.name()));
        }
        Info info = getInfo(openApiProperties);

        openAPI.setInfo(info);
        openAPI.setComponents(new Components());
        openAPI.setPaths(paths);
    }

    public static Info getInfo(OpenApiProperties openApiProperties) {
        StringBuffer titleBuffer = new StringBuffer();
        titleBuffer.append(openApiProperties.getProject()).append(":")
                .append(openApiProperties.getProjectName()).append(":")
                .append(openApiProperties.getEnv()).append(":")
                .append(openApiProperties.getEnvName()).append(":")
                .append(openApiProperties.getService()).append(":")
                .append(openApiProperties.getServiceName())
        ;
        Info info = new Info()
                .title(titleBuffer.toString())
                .description(openApiProperties.getDescription());
//                .contact(new Contact().name(openApiInfoProperties.getContactName()));
        // 无法反序列化成功
//        info.addExtension(X_OPEN_API_PROPERTIES, openApiProperties);
        info.addExtension(X_NEW_DATE, DatetimeUtil.currentTimeStr());

        // 使用相对路径打开gradle.properties文件
        info.setVersion(openApiProperties.getProjectVersion());
        info.setSummary(openApiProperties.getCloudtBootVersion());
        info.addExtension(X_PROJECT, openApiProperties.getProject());
        info.addExtension(X_PROJECT_NAME, openApiProperties.getProjectName());
        info.addExtension(X_ENV, openApiProperties.getEnv());
        info.addExtension(X_ENV_NAME, openApiProperties.getEnvName());
        info.addExtension(X_SERVICE, openApiProperties.getService());
        info.addExtension(X_SERVICE_NAME, openApiProperties.getServiceName());
        info.addExtension(X_PROJECT_VERSION, openApiProperties.getProjectVersion());
        info.addExtension(X_CLOUDT_BOOT_VERSION, openApiProperties.getCloudtBootVersion());
        info.addExtension(X_CLOUDT_BOOT_VERSION, openApiProperties.getCloudtBootVersion());
        return info;
    }

    /**
     * 处理每个控制器，获取其路径和方法信息。
     */
    public static void processController(Class<?> controllerClass, OpenAPI openAPI, Paths paths) {
        RequestMapping requestMapping = controllerClass.getAnnotation(RequestMapping.class);
        String basePath = requestMapping == null ? "" : Arrays.stream(requestMapping.value()).findFirst().orElse("");
        BusinessObject businessObject = controllerClass.getAnnotation(BusinessObject.class);

        for (Method method : controllerClass.getDeclaredMethods()) {
            if (onlyGatherBusinessOperation() && !method.isAnnotationPresent(BusinessObjectOperation.class)) {
                // 没有业务操作的忽略
                continue;
            }
            if (!isAnnotationMapping(method)) {
                continue;
            }
            Type returnType = method.getGenericReturnType();
            ResolvedSchema resolvedSchema = ModelConverters.getInstance().readAllAsResolvedSchema(returnType);
            //设置返回schema到公共变量中并返回当前接口返回对象模式
            Schema mehtodReturnSchema =
                    processMehtodRetuntComponentsGetSchema(openAPI, resolvedSchema);
            //创建操作，设置操作的业务对象 设置操作信息，设置业务对象操作信息
            Operation operation = createOperationFromApiOperation(openAPI, businessObject, method);
            //操作设置请求参数request body的请求参数
            for (Parameter parameter : method.getParameters()) {
                RequestBody requestBodyAnnotation = parameter.getAnnotation(RequestBody.class);
                if (requestBodyAnnotation != null) {
                    processRequestBody(parameter, operation, openAPI);
                }

                com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter
                        parameterV3 = createParameterForApi(parameter);
                if (parameterV3 != null) {
                    operation.addParametersItem(parameterV3);
                }
            }
            //设置返回对象
            processReturnParameter(operation, mehtodReturnSchema);
            //获取接口路径
            String path = basePath + getMethodPath(method);
            //接口路径添加到总路径清单 Paths paths
            addOperationToPathItem(path, method, operation, paths);
        }
    }

    /**
     * 通过SPI处理
     */
    public static void processBySpi(@NotNull OpenAPI openAPI) {
        // 先加载所有的SPI
        var spiList = loadSpiList();
        if (spiList.isEmpty()) {
            return;
        }

        // 通过SPI解析openApi的类
        List<Class<?>> classes = new ArrayList<>(256);
        for (OpenApiLoadStrategySPI spi : spiList) {
            if (!spi.enabled()) {
                continue;
            }
            var openApiClasses = spi.load();
            if (CollUtil.isEmpty(openApiClasses)) {
                continue;
            }
            classes.addAll(openApiClasses);
        }

        // 解析openApi
        if (classes.isEmpty()) {
            return;
        }
        for (Class<?> clazz : classes) {
            analyzeOpenApi(openAPI, clazz);
        }
    }

    /**
     * 分析OpenApi信息
     *
     * @param openAPI
     * @param clazz
     */
    public static void analyzeOpenApi(@NotNull OpenAPI openAPI, @NotNull Class<?> clazz) {
        var spiList = loadSpiList();
        if (spiList.isEmpty()) {
            return;
        }

        // 解析OpenApiInfo信息
        for (OpenApiLoadStrategySPI spi : spiList) {
            if (!spi.enabled()) {
                continue;
            }
            if (!spi.support(clazz)) {
                continue;
            }
            var openApiList = spi.analyze(clazz);
            if (CollUtil.isEmpty(openApiList)) {
                continue;
            }
            for (OpenApiLoadStrategySPI.OpenApiInfo openApiInfo : openApiList) {
                process(openAPI, openApiInfo, spi);
            }
        }
    }

    private static void process(OpenAPI openAPI, OpenApiLoadStrategySPI.OpenApiInfo openApiInfo, OpenApiLoadStrategySPI spi) {
        if (onlyGatherBusinessOperation() && openApiInfo.getBusinessObjectOperation() == null) {
            return;
        }

        Assert.notNull(openApiInfo.getMethod(), "method为空");
        Assert.notNull(openApiInfo.getUri(), "uri为空");
        // 操作信息
        var operation = createOperation(openAPI, openApiInfo);

        // 处理入参
        for (Parameter parameter : openApiInfo.getMethod().getParameters()) {
            // 请求体参数
            if (spi.isRequestBody(parameter)) {
                processRequestBody(parameter, operation, openAPI);
                continue;
            }

            // 其它参数
            var paramApi = createParameterForApi(parameter);
            if (paramApi == null) {
                continue;
            }
            operation.addParametersItem(paramApi);
        }

        // 处理返参
        var returnType = openApiInfo.getMethod().getGenericReturnType();
        ResolvedSchema resolvedSchema = modelConverters.readAllAsResolvedSchema(returnType);
        var mehtodReturnSchema = processMehtodRetuntComponentsGetSchema(openAPI, resolvedSchema);
        processReturnParameter(operation, mehtodReturnSchema);

        // 处理路径
        addOperationToPathItem(openApiInfo.getUri(), openApiInfo.getRequestMethod(), operation, openAPI.getPaths());
    }

    /**
     * 设置方法的返回参数到公共组件，并返回当前方法的返回参数模式
     */
    public static Schema processMehtodRetuntComponentsGetSchema(OpenAPI openAPI, ResolvedSchema resolvedSchema) {
        Schema mehtodReturnSchemaNew = null;
        if (resolvedSchema != null && resolvedSchema.schema != null) {
            mehtodReturnSchemaNew = resolvedSchema.schema;
        }
        if (resolvedSchema != null && resolvedSchema.referencedSchemas != null
                && resolvedSchema.referencedSchemas.size() > 0) {
            // 遍历并处理生成的 Schema
            for (Map.Entry<String, Schema> entry : resolvedSchema.referencedSchemas.entrySet()) {
                String name = entry.getKey();
                Schema schema = entry.getValue();
                // 可以根据需要添加到 OpenAPI 组件中
                openAPI.getComponents().addSchemas(name, schema);
                if (mehtodReturnSchemaNew == null) {
                    mehtodReturnSchemaNew = schema;
                }
            }
        }
        return mehtodReturnSchemaNew;
    }

    private static boolean onlyGatherBusinessOperation() {
        return true;
    }

    /**
     * 设置返回参数
     */
    private static void processReturnParameter(Operation operation,
                                               Schema mehtodReturnSchema) {
        ApiResponse apiResponse = new ApiResponse();
        apiResponse.content(new Content()
                .addMediaType(
                        org.springframework.http.MediaType.APPLICATION_JSON_VALUE,
                        new MediaType()
                                .schema(mehtodReturnSchema))
        );
        // 将 ApiResponse 添加到 Operation 的响应中
        ApiResponses apiResponses = new ApiResponses();
        apiResponses.addApiResponse("200", apiResponse); //  200 OK 响应
        operation.setResponses(apiResponses);
    }

    /**
     * 从 ApiOperation 注解创建 OpenAPI 'Operation' 对象。
     */
    private static Operation createOperationFromApiOperation(OpenAPI openAPI, BusinessObject businessObject,
                                                             Method method) {
        ApiOperation apiOperation = method.getAnnotation(ApiOperation.class);
        Operation operation = new Operation();
        operation.setSummary(apiOperation == null ? "空" : (StringUtils.isBlank(apiOperation.value()) ? "空" : apiOperation.value()));
        operation.setDescription(apiOperation == null ? "空" : (StringUtils.isBlank(apiOperation.notes()) ? "空" : apiOperation.notes()));
        operation.setDeprecated(method.isAnnotationPresent(Deprecated.class));
        operation.operationId(method.toString());

        // 业务对象
        if (businessObject != null) {
            BusinessObjectInfo businessObjectInfo = buildBusinessObjectInfo(openAPI, businessObject);
            openAPI.addTagsItem(new Tag().name(businessObject.businessType()));
        }
        // 业务操作
        BusinessObjectOperation businessObjectOperation = method.getAnnotation(BusinessObjectOperation.class);
        if (businessObjectOperation != null) {
            //设置业务对象扩展
            String businessObjectCode = BusinessObjectOperation.BUSINESS_OBJECT_NAME_DEFAULT.equals(businessObjectOperation.businessObjectType()) ?
                    (businessObject == null ? null : businessObject.businessType()) : businessObjectOperation.businessObjectType();
            Assert.hasText(businessObjectCode, () -> method.getDeclaringClass().getName() + "." + method.getName() + "()操作未明确所属业务对象");
            operation.addTagsItem(businessObjectCode);
            BusinessObjectOperationInfo businessObjectOperationInfo = BusinessObjectOperationInfo.getBusinessObjectOperationInfo(businessObjectOperation);
            businessObjectOperationInfo.setBusinessObjectType(businessObjectCode);
            operation.addExtension(X_BUSINESS_OBJECT_OPERATION, businessObjectOperationInfo);
            operation.setSummary(operation.getSummary() + "-" + businessObjectOperationInfo.getOperationStateEnum().getDescription());
            operation.addTagsItem(businessObjectOperationInfo.getOperationStateEnum().getDescription());
            operation.addTagsItem(businessObjectOperationInfo.getOperationScopeEnum().getDescription());
        }
        openAPI.addTagsItem(new Tag().name("工程版本" + openAPI.getInfo().getVersion()
                + ";架构版本" + openAPI.getInfo().getSummary()
        ));
        operation.addTagsItem("工程版本" + openAPI.getInfo().getVersion()
                + ";架构版本" + openAPI.getInfo().getSummary());
        return operation;
    }

    //返回泛型的类对象
    public static Class<?> getGenericTypeClass(Type type) {
        if (type instanceof ParameterizedType) {
            ParameterizedType parameterizedType = (ParameterizedType) type;
            Type[] actualTypeArguments = parameterizedType.getActualTypeArguments();
            if (actualTypeArguments.length > 0) {
                Type firstTypeArgument = actualTypeArguments[0];
                if (firstTypeArgument instanceof Class) {
                    return (Class<?>) firstTypeArgument;
                }
            }
        }
        return null;
    }

    //创建请求参数对象
    public static com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter
    createParameterForApi(Parameter parameter) {
        com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter parameterV3 = new com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.Parameter();
        String annotationParameterType = getAnnotationParameterType(parameter);
        if (annotationParameterType == null) {
            return null;
        }
        var schemaResolved = ModelConverters.getInstance().readAllAsResolvedSchema(parameter.getParameterizedType());
        var pType = PrimitiveType.fromType(parameter.getParameterizedType());//格式化基本类型

        Schema schema = new Schema();
        if (schemaResolved == null) {
            schema.setName(pType.getCommonName());
            schema.setTitle(pType.name());
            schema.setType(pType.name());
            schema = pType.createProperty();
        } else {
            schema = schemaResolved.schema;
        }
        parameterV3.setName(parameter.getName());
        parameterV3.setSchema(schema);
        parameterV3.setDescription(parameter.getName());
        parameterV3.setRequired(true);
        parameterV3.setIn(annotationParameterType);
        return parameterV3; // 默认为查询参数
    }

    //入参的时候判断入参对象的前缀注解是什么类型的
    public static String getAnnotationParameterType(Parameter parameter) {
        if (parameter.isAnnotationPresent(PathVariable.class)) {
            return ParameterIn.PATH.toString();
        } else if (parameter.isAnnotationPresent(RequestParam.class)) {
            return ParameterIn.QUERY.toString();
        } else if (parameter.isAnnotationPresent(RequestHeader.class)) {
            return ParameterIn.HEADER.toString();
        } else if (parameter.isAnnotationPresent(CookieValue.class)) {
            return ParameterIn.COOKIE.toString();
        } else {
            return null;
        }
    }

    public static List<BusinessObjectParamInfo> obtainBusinessObjectParamInfoList(Class<?>... entityClasses) {
        if (ArrayUtil.isEmpty(entityClasses)) {
            return Collections.emptyList();
        }

        List<BusinessObjectParamInfo> paramInfoList = new ArrayList<>(entityClasses.length);
        Set<String> entityClassNames = new HashSet<>(entityClasses.length);
        for (Class<?> entityClass : entityClasses) {
            if (entityClass == null || entityClass == Void.class) {
                continue;
            }
            if (entityClassNames.contains(entityClass.getName())) {
                continue;
            }
            entityClassNames.add(entityClass.getName());

            var paramInfo = PARAM_INFO_MAP.computeIfAbsent(entityClass, cls -> {
                BusinessObjectParamInfo info = new BusinessObjectParamInfo();
                info.setClassName(cls.getName());
                info.setTableName(obtainTableName(cls));
                info.setTableComment(obtainTableComment(cls));
                info.setParamList(obtainEntityFields(cls));
                return info;
            });

            paramInfoList.add(paramInfo);
        }
        return paramInfoList;
    }

    /**
     * 将Operation对象添加到 PathItem 中。
     */
    private static void addOperationToPathItem(String path, Method method, Operation operation, Paths paths) {
        PathItem pathItem = paths.get(path);
        if (pathItem == null) {
            pathItem = new PathItem();
        }
        setOperationBasedOnHttpMethod(method, pathItem, operation);
        paths.addPathItem(path, pathItem);
    }

    private static void addOperationToPathItem(String path, RequestMethod requestMethod, Operation operation, Paths paths) {
        PathItem pathItem = paths.get(path);
        if (pathItem == null) {
            pathItem = new PathItem();
        }
        setOperationBasedOnHttpMethod(requestMethod, pathItem, operation);
        paths.addPathItem(path, pathItem);
    }

    /**
     * 获取指定 HTTP 方法的路径。
     */
    private static String getMethodPath(Method method) {
        GetMapping getMapping = method.getAnnotation(GetMapping.class);
        if (getMapping != null && getMapping.value().length > 0) {
            return getMapping.value()[0];
        }
        // 检查 PostMapping 注解
        PostMapping postMapping = method.getAnnotation(PostMapping.class);
        if (postMapping != null && postMapping.value().length > 0) {
            return postMapping.value()[0];
        }
        // 检查 PutMapping 注解
        PutMapping putMapping = method.getAnnotation(PutMapping.class);
        if (putMapping != null && putMapping.value().length > 0) {
            return putMapping.value()[0];
        }
        // 检查 DeleteMapping 注解
        DeleteMapping deleteMapping = method.getAnnotation(DeleteMapping.class);
        if (deleteMapping != null && deleteMapping.value().length > 0) {
            return deleteMapping.value()[0];
        }
        PatchMapping patchMapping = method.getAnnotation(PatchMapping.class);
        if (patchMapping != null && patchMapping.value().length > 0) {
            return patchMapping.value()[0];
        }
        RequestMapping requestMapping = method.getAnnotation(RequestMapping.class);
        if (requestMapping != null && requestMapping.value().length > 0) {
            return requestMapping.value()[0];
        }
        // 默认返回空字符串
        return "";
    }

    private static String generateHash(String methodName) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hashBytes = digest.digest(methodName.getBytes());
            StringBuilder hexString = new StringBuilder();
            for (byte b : hashBytes) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-256 algorithm not found", e);
        }
    }

    /**
     * 根据方法的 HTTP 注解设置相应的操作类型。
     */
    private static void setOperationBasedOnHttpMethod(Method method, PathItem pathItem, Operation operation) {
        // 根据方法注解设置对应的HTTP操作
        if (method.isAnnotationPresent(GetMapping.class)) {
            pathItem.setGet(operation);
        } else if (method.isAnnotationPresent(PostMapping.class)) {
            pathItem.setPost(operation);
        } else if (method.isAnnotationPresent(PutMapping.class)) {
            pathItem.setPut(operation);
        } else if (method.isAnnotationPresent(DeleteMapping.class)) {
            pathItem.setDelete(operation);
        } else if (method.isAnnotationPresent(PatchMapping.class)) {
            pathItem.setPatch(operation);
        } else if (method.isAnnotationPresent(RequestMapping.class)) {
            var requestMapping = method.getAnnotation(RequestMapping.class);
            var methods = requestMapping.method();
            if (methods.length > 0) {
                var methodMapping = methods[0];
                switch (methodMapping) {
                    case GET:
                        pathItem.setGet(operation);
                        break;
                    case POST:
                        pathItem.setPost(operation);
                        break;
                    case PUT:
                        pathItem.setPut(operation);
                        break;
                    case DELETE:
                        pathItem.setDelete(operation);
                        break;
                    case PATCH:
                        pathItem.setPatch(operation);
                        break;
                    case HEAD:
                        pathItem.setHead(operation);
                        break;
                    case TRACE:
                        pathItem.setHead(operation);
                        break;
                    case OPTIONS:
                        pathItem.setOptions(operation);
                        break;
                    default:
                        break;
                }
            }
        }
        processOperation(operation);
    }

    private static void setOperationBasedOnHttpMethod(RequestMethod requestMethod, PathItem pathItem, Operation operation) {
        switch (requestMethod) {
            case GET:
                pathItem.setGet(operation);
                break;
            case POST:
                pathItem.setPost(operation);
                break;
            case PUT:
                pathItem.setPut(operation);
                break;
            case PATCH:
                pathItem.setPatch(operation);
                break;
            case DELETE:
                pathItem.setDelete(operation);
                break;
        }

        processOperation(operation);
    }

    /**
     * 根据方法的 HTTP 注解设置相应的操作类型。
     */
    private static Boolean isAnnotationMapping(Method method) {
        // 根据方法注解设置对应的HTTP操作
        if (method.isAnnotationPresent(GetMapping.class)) {
            return true;
        }
        if (method.isAnnotationPresent(PostMapping.class)) {
            return true;
        }
        if (method.isAnnotationPresent(PutMapping.class)) {
            return true;
        }
        if (method.isAnnotationPresent(DeleteMapping.class)) {
            return true;
        }
        if (method.isAnnotationPresent(PatchMapping.class)) {
            return true;
        }
        if (method.isAnnotationPresent(RequestMapping.class)) {
            return true;
        }

        return false;
    }

    private static void processRequestBody(Parameter parameter, Operation operation, OpenAPI openAPI) {
        Class<?> paramType = parameter.getType();

        Schema returnSchema;
        // 如果是集合类型 需要单独设置
        if (paramType.isArray() || Collection.class.isAssignableFrom(paramType)) {
            Type parameterizedType = parameter.getParameterizedType();
            // 转换为Schema对象
            ResolvedSchema resolvedSchema = ModelConverters.getInstance()
                    .readAllAsResolvedSchema(parameterizedType);
            if (resolvedSchema == null) {
                return;
            }
            Map<String, Schema> schemaMap = resolvedSchema.referencedSchemas;
            // 集合类型的泛型类获取泛型的模式
            Schema pSchema = schemaMap.get(getGenericTypeClass(parameterizedType).getSimpleName());
            returnSchema = new ArraySchema().name(paramType.getName())
                    .items(pSchema)
                    .description(parameter.getParameterizedType().getTypeName());
            // 将入参存入公共参数
            for (Map.Entry<String, Schema> entry : schemaMap.entrySet()) {
                String name = entry.getKey();
                Schema schema = entry.getValue();
                openAPI.getComponents().addSchemas(name, schema);
            }
        } else if (Map.class.isAssignableFrom(paramType)) {
            returnSchema = new MapSchema();
            returnSchema.setAdditionalProperties(true);
            returnSchema.description(parameter.getParameterizedType().getTypeName());
            returnSchema.name(paramType.getSimpleName());
        } else {
            //如果不是集合类型 表示是对象
            ResolvedSchema resolvedSchema = ModelConverters.getInstance()
                    .readAllAsResolvedSchema(paramType);
            if (resolvedSchema == null) {
                return;
            }
            Map<String, Schema> schemaMap = resolvedSchema.referencedSchemas;
            if (schemaMap == null) {
                return;
            }
            if (parameter.getParameterizedType() instanceof ParameterizedType) {
                log.error("不是数组有泛型");
            }
            returnSchema = schemaMap.get(paramType.getSimpleName());
            for (Map.Entry<String, Schema> entry : schemaMap.entrySet()) {
                String name = entry.getKey();
                Schema schema = entry.getValue();
                openAPI.getComponents().addSchemas(name, schema);
            }
        }
        com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.RequestBody requestBody
                = new com.elitescloud.boot.swagger.openapi.swagger3.models.parameters.RequestBody();
        MediaType mediaType
                = new MediaType();

        // 设置请求体内容
        Content content = new Content();
        content.addMediaType(org.springframework.http.MediaType.APPLICATION_JSON_VALUE,
                mediaType.schema(returnSchema));
        requestBody.content(content);
        operation.setRequestBody(requestBody);
    }

    private static Operation createOperation(OpenAPI openAPI, OpenApiLoadStrategySPI.OpenApiInfo openApiInfo) {
        var method = openApiInfo.getMethod();
        ApiOperation apiOperation = method.getAnnotation(ApiOperation.class);
        Operation operation = new Operation();
        operation.setSummary(apiOperation == null ? "空" : (StringUtils.isBlank(apiOperation.value()) ? "空" : apiOperation.value()));
        operation.setDescription(apiOperation == null ? "空" : (StringUtils.isBlank(apiOperation.notes()) ? "空" : apiOperation.notes()));
        operation.setDeprecated(method.isAnnotationPresent(Deprecated.class));
        operation.operationId(method.toString());

        openAPI.addTagsItem(new Tag().name("工程版本" + openAPI.getInfo().getVersion()
                + ";架构版本" + openAPI.getInfo().getSummary()
        ));
        operation.addTagsItem("工程版本" + openAPI.getInfo().getVersion()
                + ";架构版本" + openAPI.getInfo().getSummary());

        // 业务对象
        BusinessObjectInfo businessObjectInfo = null;
        String businessObjectType = null;
        if (openApiInfo.getBusinessObject() != null) {
            var businessObject = openApiInfo.getBusinessObject();
            businessObjectInfo = buildBusinessObjectInfo(openAPI, businessObject);
            businessObjectType = businessObject.businessType();
        } else if (org.springframework.util.StringUtils.hasText(openApiInfo.getBusinessObjectType())) {
            businessObjectType = openApiInfo.getBusinessObjectType();
        }
        if (!org.springframework.util.StringUtils.hasText(businessObjectType)) {
            return operation;
        }
        openAPI.addTagsItem(new Tag().name(businessObjectType));
        operation.addTagsItem(businessObjectType);

        // 业务操作
        BusinessObjectOperationInfo businessObjectOperationInfo = buildBusinessObjectOperation(openApiInfo);
        if (businessObjectOperationInfo != null) {
            if (BusinessObjectOperation.BUSINESS_OBJECT_NAME_DEFAULT.equals(businessObjectOperationInfo.getBusinessObjectType())) {
                businessObjectOperationInfo.setBusinessObjectType(businessObjectType);
            }
            operation.addExtension(X_BUSINESS_OBJECT_OPERATION, businessObjectOperationInfo);

            StringBuffer summary = new StringBuffer();
            summary.append(operation.getSummary()).append("-")
                    .append(businessObjectOperationInfo.getOperationStateEnum().getDescription());
            operation.setSummary(summary.toString());
            operation.addTagsItem(businessObjectOperationInfo.getOperationStateEnum().getDescription());
            operation.addTagsItem(businessObjectOperationInfo.getOperationScopeEnum().getDescription());
        }
        return operation;
    }

    private static BusinessObjectOperationInfo buildBusinessObjectOperation(OpenApiLoadStrategySPI.OpenApiInfo openApiInfo) {
        if (openApiInfo.getBusinessObjectOperation() != null) {
            return BusinessObjectOperationInfo.getBusinessObjectOperationInfo(openApiInfo.getBusinessObjectOperation());
        }

        BusinessObjectOperationInfo businessObjectOperationInfo = new BusinessObjectOperationInfo();
        businessObjectOperationInfo.setEnabled(true);
        businessObjectOperationInfo.setBusinessObjectType(openApiInfo.getBusinessObjectType());
        businessObjectOperationInfo.setOperationCode(openApiInfo.getBusinessOperationCode());
        businessObjectOperationInfo.setOperationType(openApiInfo.getOperationType());
        businessObjectOperationInfo.setOperationDescription(openApiInfo.getOperationDescription());
        businessObjectOperationInfo.setOperationScopeEnum(OperationScopeEnum.PRIVATE_DOMAIN);
        businessObjectOperationInfo.setOperationStateEnum(OperationStateEnum.RELEASED);
        businessObjectOperationInfo.setDataPermissionEnabled(openApiInfo.isDataPermissionEnabled());
        businessObjectOperationInfo.setFieldPermissionAutoFilter(openApiInfo.isDataPermissionEnabled());
        businessObjectOperationInfo.setPermissionRef(openApiInfo.getPermissionRef());

        return businessObjectOperationInfo;
    }

    private static List<OpenApiLoadStrategySPI> loadSpiList() {
        if (!spiList.isEmpty()) {
            return Collections.unmodifiableList(spiList);
        }

        for (OpenApiLoadStrategySPI spi : ServiceLoader.load(OpenApiLoadStrategySPI.class)) {
            spiList.add(spi);
        }
        return Collections.unmodifiableList(spiList);
    }

    @SuppressWarnings("unchecked")
    private static BusinessObjectInfo buildBusinessObjectInfo(OpenAPI openAPI, BusinessObject businessObject) {
        var extensions = openAPI.getExtensions();
        if (extensions == null) {
            extensions = new HashMap<>(8);
            openAPI.extensions(extensions);
        }
        Map<String, BusinessObjectInfo> businessObjectMap = (Map<String, BusinessObjectInfo>) extensions.computeIfAbsent(X_BUSINESS_OBJECTS, k -> new HashMap<>());

        BusinessObjectInfo businessObjectInfo = businessObjectMap.get(businessObject.businessType());
        if (businessObjectInfo == null || businessObject.businessDoClass() != null) {
            // 优先取指定了实体类的业务对象
            businessObjectInfo = BusinessObjectInfo.getBusinessObjectInfo(businessObject);
            businessObjectInfo.setParamInfoList(obtainBusinessObjectParamInfoList(businessObject.businessDoClass()));
            businessObjectMap.put(businessObject.businessType(), businessObjectInfo);
        }
        return businessObjectInfo;
    }

    private static String obtainTableName(Class<?> entityClass) {
        var table = entityClass.getAnnotation(Table.class);
        if (table != null && org.springframework.util.StringUtils.hasText(table.name())) {
            return table.name();
        }

        var tableHibernate = entityClass.getAnnotation(org.hibernate.annotations.Table.class);
        if (tableHibernate != null && org.springframework.util.StringUtils.hasText(tableHibernate.appliesTo())) {
            return tableHibernate.appliesTo();
        }
        return null;
    }

    private static String obtainTableComment(Class<?> entityClass) {
        var comment = entityClass.getAnnotation(Comment.class);
        if (comment != null && org.springframework.util.StringUtils.hasText(comment.value())) {
            return comment.value();
        }


        var commentHibernate = entityClass.getAnnotation(org.hibernate.annotations.Comment.class);
        if (commentHibernate != null && org.springframework.util.StringUtils.hasText(commentHibernate.value())) {
            return commentHibernate.value();
        }

        var tableHibernate = entityClass.getAnnotation(org.hibernate.annotations.Table.class);
        if (tableHibernate != null && org.springframework.util.StringUtils.hasText(tableHibernate.comment())) {
            return tableHibernate.comment();
        }
        return null;
    }

    private static List<BusinessObjectParamInfo.Param> obtainEntityFields(Class<?> entityClass) {
        var fields = ReflectUtil.getFields(entityClass);
        if (ArrayUtil.isEmpty(fields)) {
            return Collections.emptyList();
        }

        // 获取基类字段
        Set<String> baseFields = obtainBaseFields(entityClass);

        var isEntityClass = entityClass.isAnnotationPresent(Entity.class) || entityClass.isAnnotationPresent(org.hibernate.annotations.Entity.class);
        return Arrays.stream(fields)
                .filter(field -> {
                    if (!baseFields.isEmpty() && baseFields.contains(field.getName())) {
                        return true;
                    }

                    return isNormalField(field, isEntityClass);
                })
                .map(field -> {
                    BusinessObjectParamInfo.Param param = new BusinessObjectParamInfo.Param();
                    param.setFieldName(field.getName());
                    param.setColumnName(convertColumnName(field));
                    param.setFieldComment(obtainFieldComment(field));
                    var javaType = ObjectUtil.defaultIfNull(field.getType().getComponentType(), field.getType());
                    param.setJavaType(javaType.getName());
                    param.setSingle(!field.getType().isArray() && !Collection.class.isAssignableFrom(field.getType()));

                    // 枚举类
                    param.setEnumeration(Enum.class.isAssignableFrom(javaType));
                    if (param.isEnumeration()) {
                        var enumAnnos = field.getAnnotation(Enumerated.class);
                        if (enumAnnos != null) {
                            param.setEnumType(enumAnnos.value().name());
                        } else {
                            param.setEnumType(EnumType.ORDINAL.name());
                        }
                    }

                    // 关联业务对象
                    var businessParam = field.getAnnotation(BusinessParam.class);
                    if (businessParam != null) {
                        if (StringUtils.isNotBlank(businessParam.relatedBusinessObject())) {
                            var tempArray = businessParam.relatedBusinessObject().split(":");
                            param.setRelatedBusinessObject(tempArray[0]);
                            param.setRelatedField(businessParam.relatedField());
                        }
                        param.setValues(businessParam.values());
                        if (StringUtils.isNotBlank(businessParam.userType())) {
                            param.setUserType(businessParam.userType());
                        }
                        param.setDataPermissionEnabled(businessParam.dataPermissionEnabled());
                        param.setFieldPermissionEnabled(businessParam.fieldPermissionEnabled());
                    }
                    if (!baseFields.isEmpty()) {
                        param.setBaseField(baseFields.contains(field.getName()));
                    }
                    param.setDeprecated(field.isAnnotationPresent(Deprecated.class));

                    return param;
                }).collect(Collectors.toList());
    }

    private static Set<String> obtainBaseFields(Class<?> clazz) {
        if (!BaseModel.class.isAssignableFrom(clazz)) {
            return Collections.emptySet();
        }
        var fields = ReflectUtil.getFields(BaseModel.class);
        if (ArrayUtil.isEmpty(fields)) {
            return Collections.emptySet();
        }

        return Arrays.stream(fields)
                .filter(t -> isNormalField(t, true)).map(Field::getName)
                .collect(Collectors.toSet());
    }

    private static boolean isNormalField(Field t, boolean isEntity) {
        if (Modifier.isTransient(t.getModifiers())) {
            return false;
        }
        if (t.isAnnotationPresent(Transient.class) || t.isAnnotationPresent(org.springframework.data.annotation.Transient.class)) {
            return false;
        }
        if (Modifier.isStatic(t.getModifiers())) {
            return false;
        }
        if (isEntity) {
            return t.isAnnotationPresent(Column.class);
        }

        return true;
    }

    private static String convertColumnName(Field field) {
        var column = field.getAnnotation(Column.class);
        if (column != null && org.springframework.util.StringUtils.hasText(column.name())) {
            return column.name();
        }
        return NamingCase.toUnderlineCase(field.getName());
    }

    private static String obtainFieldComment(Field field) {
        var comment = field.getAnnotation(Comment.class);
        if (comment != null && org.springframework.util.StringUtils.hasText(comment.value())) {
            return comment.value();
        }

        var commentHibernate = field.getAnnotation(org.hibernate.annotations.Comment.class);
        if (commentHibernate != null && org.springframework.util.StringUtils.hasText(commentHibernate.value())) {
            return commentHibernate.value();
        }

        var apiModelProperty = field.getAnnotation(ApiModelProperty.class);
        if (apiModelProperty != null && org.springframework.util.StringUtils.hasText(apiModelProperty.value())) {
            return apiModelProperty.value();
        }
        return null;
    }
}
