package com.elitescloud.boot.swagger.feignapi;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.swagger.openapi.common.spi.OpenApiLoadStrategySPI;
import com.elitescloud.boot.swagger.openapi.service.impl.OpenFeignApiServiceUtil;
import com.elitescloud.boot.util.ArrayUtil;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ProxyUtil;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author : chen.niu
 * @description :
 * @date : 2024-07-17 15:09
 */
public class OpenApiLoadStrategyRpc implements OpenApiLoadStrategySPI {

    @Override
    public List<Class<?>> load() {
        // 获取含有DocumentedFeignApi注解的bean
        Map<String, Object> beansWithAnnotation = SpringContextHolder.getBeansByAnnotation(DocumentedFeignApi.class);
        if (beansWithAnnotation.isEmpty()) {
            return Collections.emptyList();
        }

        // 获取其含有DocumentedFeignApi注解的接口
        return beansWithAnnotation.values().stream()
                .flatMap(t -> {
                    Class<?>[] interfaces = ProxyUtil.getTargetClass(t).getInterfaces();
                    if (ArrayUtil.isEmpty(interfaces)) {
                        return Stream.empty();
                    }
                    return Arrays.stream(interfaces).filter(tt -> tt.isAnnotationPresent(DocumentedFeignApi.class));
                }).distinct().collect(Collectors.toList());
    }

    @Override
    public List<OpenApiInfo> analyze(Class<?> clazz) {
        FeignClient requestMapping = clazz.getAnnotation(FeignClient.class);
        if (requestMapping == null) {
            return Collections.emptyList();
        }
        String basePath = requestMapping.path();
        DocumentedFeignApi documentedFeignApi = clazz.getAnnotation(DocumentedFeignApi.class);

        List<OpenApiInfo> openApiInfos = new ArrayList<>();
        for (Method method : clazz.getDeclaredMethods()) {
            if (!OpenFeignApiServiceUtil.isAnnotationMapping(method)) {
                continue;
            }
            DocumentedFeignApi methodAnnotation =  method.getAnnotation(DocumentedFeignApi.class);

            String methodPath = OpenFeignApiServiceUtil.getMethodPath(method);
            OpenApiInfo openApiInfo = new OpenApiInfo(method,
                    getOperationBasedOnHttpMethod(method)
                    , basePath + methodPath);
            openApiInfo.setApiType(2);
            openApiInfo.setOperationGroup(documentedFeignApi.description());
            openApiInfo.setDetail(methodAnnotation!=null?methodAnnotation.description():"");
            openApiInfo.setOperationDescription(clazz.getName()+"-"+method.getName());

            // 获取方法的返回类型并转换为JSON
            String returnTypeJson = JSONUtil.toJsonString(
                    method.getGenericReturnType().getTypeName()
            );
            openApiInfo.setReturnTypeJson(returnTypeJson);
            // 获取方法的参数类型并转换为JSON

            List<String> parameterTypes = new ArrayList<>();
            for (Type parameterType : method.getGenericParameterTypes()) {
                parameterTypes.add(parameterType.getTypeName());
            }
            String parameterTypesJson = JSONUtil.toJsonString(parameterTypes);
            openApiInfo.setParameterTypesJson(parameterTypesJson);

            openApiInfos.add(openApiInfo);
        }

        return openApiInfos;
    }


    /**
     * 根据方法的 HTTP 注解设置相应的操作类型。
     */
    private static RequestMethod getOperationBasedOnHttpMethod(Method method) {
        // 根据方法注解设置对应的HTTP操作
        if (method.isAnnotationPresent(GetMapping.class)) {
            return RequestMethod.GET;
        } else if (method.isAnnotationPresent(PostMapping.class)) {
            return RequestMethod.POST;
        } else if (method.isAnnotationPresent(PutMapping.class)) {
            return RequestMethod.PUT;
        } else if (method.isAnnotationPresent(DeleteMapping.class)) {
            return RequestMethod.DELETE;
        } else if (method.isAnnotationPresent(PatchMapping.class)) {
            return RequestMethod.PATCH;
        } else if (method.isAnnotationPresent(RequestMapping.class)) {
            var requestMapping = method.getAnnotation(RequestMapping.class);
            var methods = requestMapping.method();
            if (methods.length > 0) {
                var methodMapping = methods[0];
                switch (methodMapping) {
                    case GET:
                        return RequestMethod.GET;
                    case POST:
                        return RequestMethod.POST;
                    case PUT:
                        return RequestMethod.PUT;
                    case DELETE:
                        return RequestMethod.DELETE;
                    case PATCH:
                        return RequestMethod.PATCH;
                    case HEAD:
                        return RequestMethod.HEAD;
                    case TRACE:
                        return RequestMethod.TRACE;
                    case OPTIONS:
                        return RequestMethod.OPTIONS;
                    default:
                        throw new RuntimeException("未找到RequestMapping注解" + methodMapping);
                }
            } else {
                throw new RuntimeException("未找到RequestMapping注解methodMapping");
            }
        } else {
            throw new RuntimeException("未找到HTTP注解");
        }
    }

    @Override
    public boolean isRequestBody(Parameter parameter) {
        return false;
    }
}
