package com.elitescloud.boot.task.delay.support.redis;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.task.delay.common.DelayTaskHandler;
import com.elitescloud.boot.task.delay.support.DelayTaskListener;
import com.elitescloud.boot.task.delay.support.TaskHelper;
import com.elitescloud.cloudt.core.task.delay.TaskInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import java.io.Serializable;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/25
 */
@Slf4j
public class RedisDelayTaskListener<T extends Serializable> implements DelayTaskListener<T>, SchedulingConfigurer {

    private final RedisTemplate<Object, Object> redisTemplate;
    /**
     * 任务组
     */
    private final String taskGroup;
    /**
     * 刷新时间 间隔
     */
    private final Duration interval;
    private Map<String, Set<DelayTaskHandler<T>>> taskHandlers;
    private ScheduledTaskRegistrar scheduledTaskRegistrar;

    private static final ZoneOffset ZONE_OFFSET = ZoneOffset.ofHours(8);
    private final String taskDataPrefix;

    public RedisDelayTaskListener(RedisTemplate<Object, Object> redisTemplate, String taskGroup, Duration interval) {
        this.redisTemplate = redisTemplate;
        this.taskGroup = taskGroup;
        this.interval = interval;
        this.taskDataPrefix = TaskHelper.packTaskDataKey(taskGroup);
    }

    @Override
    public void registerHandler(Map<String, Set<DelayTaskHandler<T>>> delayTaskHandlers) {
        log.info("Redis获取延时任务时间间隔：{}s", interval.toSeconds());
        this.taskHandlers = delayTaskHandlers;
        registerTask();
    }

    @Override
    public void configureTasks(@NonNull ScheduledTaskRegistrar taskRegistrar) {
        this.scheduledTaskRegistrar = taskRegistrar;
        registerTask();
    }

    private void registerTask() {
        if (CollUtil.isEmpty(this.taskHandlers)) {
            return;
        }
        if (this.scheduledTaskRegistrar == null) {
            return;
        }

        scheduledTaskRegistrar.addFixedDelayTask(this::pullTopic, interval.toMillis());
    }

    private void pullTopic() {
        try {
            execPull();
        } catch (Exception e) {
            log.info("Redis获取延时任务失败", e);
        }
    }

    private void execPull() throws Exception {
        Set<Object> taskList = redisTemplate.opsForZSet().rangeByScore(taskGroup, 0, LocalDateTime.now().toEpochSecond(ZONE_OFFSET));
        if (CollUtil.isNotEmpty(taskList)) {
            taskList.forEach(key ->
                    {
                        Object value = redisTemplate.opsForHash().get(taskDataPrefix, key);
                        if (value != null) {
                            redisTemplate.opsForHash().delete(taskDataPrefix, key);

                            // 取值并消费
                            TaskInfo<T> taskInfo = (TaskInfo<T>) value;
                            var handlers = taskHandlers.get(taskInfo.getTaskType());
                            if (CollUtil.isEmpty(handlers)) {
                                log.warn("未找到延时任务【{}】的处理器", taskInfo.getTaskType());
                            } else {
                                handlers.forEach(service -> {
                                    log.info("延时任务【{}】的消息：{}", service.getClass().getName(), taskInfo);
                                    try {
                                        service.execute(taskInfo);
                                    } catch (Exception e) {
                                        log.info("延时任务处理异常：", e);
                                    }
                                });
                            }
                        }

                        // 移除消息
                        redisTemplate.opsForZSet().remove(taskGroup, key);
                    }
            );
        }
    }
}
