package com.elitescloud.boot.task.delay.support.redis;

import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.task.delay.support.AbstractTaskSender;
import com.elitescloud.boot.task.delay.support.TaskHelper;
import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/25
 */
@Slf4j
public class RedisDelayTaskSender extends AbstractTaskSender {

    private final RedisTemplate<Object, Object> redisTemplate;
    private final TenantDataIsolateProvider tenantDataIsolateProvider;
    private final TenantClientProvider tenantClientProvider;
    private final String taskGroup;
    private final String taskDataPrefix;

    private static final ZoneOffset ZONE_OFFSET = ZoneOffset.ofHours(8);

    public RedisDelayTaskSender(RedisTemplate<Object, Object> redisTemplate,
                                TenantClientProvider tenantClientProvider,
                                TenantDataIsolateProvider tenantDataIsolateProvider,
                                String taskGroup) {
        this.redisTemplate = redisTemplate;
        this.tenantDataIsolateProvider = tenantDataIsolateProvider;
        this.tenantClientProvider = tenantClientProvider;
        this.taskGroup = taskGroup;
        this.taskDataPrefix = TaskHelper.packTaskDataKey(taskGroup);
    }

    @Override
    public <T extends Serializable> void send(String taskType, String taskKey, T payload, LocalDateTime delayTime) {
        LocalDateTime delay = delayTime == null ? LocalDateTime.now() : delayTime;

        // 前置处理
        beforeSend(taskType, taskKey, payload, delay);
        var key = TaskHelper.packTaskKey(taskType, taskKey);
        var tenant = tenantClientProvider.getSessionTenant();
        var tenantId = tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();

        try {
            tenantDataIsolateProvider.byDefaultDirectly(() -> {
                redisTemplate.opsForZSet().add(taskGroup, key, delay.toEpochSecond(ZONE_OFFSET));
                redisTemplate.opsForHash().put(taskDataPrefix, key, super.buildTask(
                        tenantId, taskType, taskKey, payload, delayTime
                ));
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("发送任务失败", e);
        }
    }

    @Override
    public <T extends Serializable> void send(String taskType, String taskKey, T payload) {
        send(taskType, taskKey, payload, null);
    }

    @Override
    public void cancel(String taskType, String taskKey) {
        // 前置处理
        beforeCancel(taskType, taskKey);

        var key = TaskHelper.packTaskKey(taskType, taskKey);

        try {
            redisTemplate.opsForZSet().remove(taskGroup, key);
            redisTemplate.opsForHash().delete(taskDataPrefix, key);
        } catch (Exception e) {
            throw new BusinessException("取消任务失败", e);
        }
    }
}
