package com.elitescloud.boot.task.delay.support;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.task.delay.common.TaskInfo;
import com.elitescloud.cloudt.core.task.delay.DelayTaskSender;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/08/25
 */
@Slf4j
public abstract class AbstractTaskSender implements DelayTaskSender {

    protected static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * 发送前的处理
     *
     * @param taskType  任务类型
     * @param taskKey   任务标识
     * @param payload   负载数据
     * @param delayTime 延迟处理时间
     * @param <T>       数据类型
     */
    protected <T extends Serializable> void beforeSend(String taskType, String taskKey, T payload, LocalDateTime delayTime) {
        Assert.notBlank(taskType, "任务类型为空");
        Assert.notBlank(taskKey, "任务标识为空");
        Assert.notNull(payload, "负载数据为空");

        log.info("发送延时任务：{}-{}：{}，预计处理时间：{}", taskType, taskKey, payload, formatTime(delayTime));
    }

    /**
     * 取消前的处理
     *
     * @param taskType 任务类型
     * @param taskKey  任务标识
     */
    protected void beforeCancel(String taskType, String taskKey) {
        Assert.notBlank(taskType, "任务类型为空");
        Assert.notBlank(taskKey, "任务标识为空");

        log.info("取消延时任务：{}-{}", taskKey, taskKey);
    }

    public <T extends Serializable> TaskInfo<T> buildTask(Long tenantId, String taskType, String taskKey, T payload, LocalDateTime delayTime) {
        TaskInfo<T> taskInfo = new com.elitescloud.cloudt.core.task.delay.TaskInfo<>();
        taskInfo.setTenantId(tenantId);
        taskInfo.setTaskType(taskType);
        taskInfo.setTaskKey(taskKey);
        taskInfo.setPayload(payload);
        taskInfo.setTimeCreate(LocalDateTime.now());
        taskInfo.setDelayTime(delayTime);

        return taskInfo;
    }

    /**
     * 格式化时间
     *
     * @param dateTime 时间
     * @return 时间字符串
     */
    protected String formatTime(LocalDateTime dateTime) {
        if (dateTime == null) {
            return "";
        }
        return DATE_TIME_FORMATTER.format(dateTime);
    }
}
