package com.elitescloud.boot.tenant.datasources;

import com.baomidou.dynamic.datasource.DynamicRoutingDataSource;


import com.baomidou.dynamic.datasource.creator.DefaultDataSourceCreator;
import com.baomidou.dynamic.datasource.spring.boot.autoconfigure.DataSourceProperty;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;

import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author : chen.niu
 * @description : 数据源初始化
 * @date : 2023/9/25 15:47
 */

@Order(Ordered.HIGHEST_PRECEDENCE)
@Slf4j
public class StartupRunner implements InitializingBean {


    private final DefaultDataSourceCreator dataSourceCreator;

    private final DataSource dataSource;


    private final SpringDatasourceDynamicTenantProperties datasourceProperties;

    public StartupRunner(DefaultDataSourceCreator dataSourceCreator, DataSource dataSource,
                         SpringDatasourceDynamicTenantProperties datasourceProperties) {
        this.dataSourceCreator = dataSourceCreator;
        this.dataSource = dataSource;
        this.datasourceProperties = datasourceProperties;
    }




    @Override
    public void afterPropertiesSet() {
        try {
            if (datasourceProperties.isTenantDataSourceSwitch()) {
                String appName = datasourceProperties.getAppName();
                //查询共享库中租户的数据源信息
                log.info("多租户数据源初始化执行-开始");
                List<SysDatabaseServiceTenantDTO> tenantDataSources = querySysDatabase();
                tenantDataSources.forEach(tenantDataSource -> {
                    if (StringUtils.hasText(tenantDataSource.getServiceCode())) {
                        //判断配置数据源服务编码和当前编码一致
                        if (tenantDataSource.getServiceCode().equals(appName)) {

                            log.info("租户数据源开始初始化启动：租户编码：{}，{}", tenantDataSource.getTenantCode(), tenantDataSource.getServiceCode());
                            DataSourceProperty dataSourceProperty = getDataSourceProperty(tenantDataSource);
                            DynamicRoutingDataSource ds = (DynamicRoutingDataSource) dataSource;
                            DataSource dataSource = dataSourceCreator.createDataSource(dataSourceProperty);
                            ds.addDataSource(appName + "_" + tenantDataSource.getTenantCode(), dataSource);
                            log.info("租户数据源开始初始化启动完毕：租户编码：{}，{}", tenantDataSource.getTenantCode(), tenantDataSource.getServiceCode());
                        } else {
                            log.info("非本服务数据源跳过初始化：{}!={}", appName, tenantDataSource.getServiceCode());
                        }
                    } else {
                        log.error("数据源没有配置服务编码{}", tenantDataSource.getServiceCode());
                    }
                });
                log.info("多租户数据源执行-结束");
            } else {
                log.info("未开启多租户多数据源，采用多租户单数据源逻辑隔离: {}", datasourceProperties);
            }
        } catch (Exception e) {
            System.exit(1);
        }
    }

    /**
     * 私有方法：返回数据源配置信息对象
     **/
    private static DataSourceProperty getDataSourceProperty(SysDatabaseServiceTenantDTO tenantDataSource) {
        Map<String, String> replacements = new HashMap<>();
        replacements.put("dbsNetaddress", tenantDataSource.getDbsNetaddress());
        replacements.put("dbsDbPort", tenantDataSource.getDbsDbPort());
        replacements.put("dbsDbName",tenantDataSource.getDbsName());
//        replacements.put("dbsDriverClass",tenantDataSource.getDbsDriverClass());
        String dbsUrl = replaceNamedPlaceholders(tenantDataSource.getDbsUrl(), replacements);

        DataSourceProperty dataSourceProperty = new DataSourceProperty();
        dataSourceProperty.setDriverClassName(tenantDataSource.getDbsDriverClass());
        dataSourceProperty.setUrl(dbsUrl);
        dataSourceProperty.setUsername(tenantDataSource.getDbsUsername());
        dataSourceProperty.setPassword(tenantDataSource.getDbsPassword());

        return dataSourceProperty;
    }

        //占位符替换
    public static String replaceNamedPlaceholders(String str, Map<String, String> replacements) {
        for (Map.Entry<String, String> entry : replacements.entrySet()) {
            str = str.replace("{" + entry.getKey() + "}", entry.getValue());
        }
        return str;
    }


    private List<SysDatabaseServiceTenantDTO> querySysDatabase() throws SQLException {
        List<SysDatabaseServiceTenantDTO> resultList = new ArrayList<>();

        if (dataSource.getConnection() != null) {
            String sql = "SELECT * FROM sys_database_source sd where sd.deleteFlag=0";
            try (Statement statement = dataSource.getConnection().createStatement();
                 ResultSet rs = statement.executeQuery(sql)) {
                while (rs.next()) {
                    SysDatabaseServiceTenantDTO dto = new SysDatabaseServiceTenantDTO();
                    dto.setDbsName(rs.getString("dbs_name"))
                            .setDbsDriverClass(rs.getString("dbs_driver_class"))
                            .setDbsUrl(rs.getString("dbs_url"))
                            .setDbsDbName(rs.getString("dbs_db_name"))
                            .setDbsDbPort(rs.getString("dbs_db_port"))
                            .setDbsUsername(rs.getString("dbs_username"))
                            .setDbsNetaddress(rs.getString("dbs_netaddress"))
                            .setDbsPassword(rs.getString("dbs_password"))
                            .setServiceCode(rs.getString("service_code"))
                            .setTenantCode(rs.getString("tenant_code"))
                            .setTenantOrgId(rs.getLong("tenant_org_id"));
                    resultList.add(dto);
                }
            } catch (Exception e) {
               log.error("多租户多数据源初始化异常："+e.getMessage());
            } finally {
                try {
                    dataSource.getConnection().close();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return resultList;
    }
}
