package com.elitescloud.boot.tenant.client.support.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.flyway.common.FlywayBuilder;
import com.elitescloud.boot.flyway.common.FlywayHelper;
import com.elitescloud.boot.tenant.client.TenantClientProperties;
import com.elitescloud.boot.tenant.client.common.AbstractTenantDatasourceProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.boot.tenant.client.common.TenantDatabaseRpcProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.TenantConstant;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.context.util.database.SqlUtil;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

/**
 * 租户数据库变更接收者.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/25
 */
@DubboService
@ResponseBody
@RequestMapping(value = TenantConstant.URI_PATH_DB_PREFIX, produces = MediaType.APPLICATION_JSON_VALUE)
@Log4j2
public class TenantDatabaseRpcProviderImpl implements TenantDatabaseRpcProvider {

    @Autowired(required = false)
    private FlywayBuilder flywayBuilder;
    @Autowired(required = false)
    private FlywayHelper flywayHelper;
    @Autowired(required = false)
    private TenantClientProperties tenantClientProperties;
    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Override
    public ApiResult<Boolean> schemaCreate(String schema) {
        if (!TenantClient.enabledTenant()) {
            log.warn("未启用租户功能");
            return ApiResult.ok(true, "未启用租户功能");
        }
        if (tenantClientProperties.getIsolateStrategy() == TenantIsolateStrategy.NONE
                || tenantClientProperties.getIsolateStrategy() == TenantIsolateStrategy.FIELD) {
            log.warn("当前策略{}无需创建schema", tenantClientProperties.getIsolateStrategy());
            return ApiResult.ok(true, "当前隔离策略无需创建");
        }
        log.info("创建新租户schema：{}", schema);

        try {
            flywayHelper.migrate(() -> flywayBuilder.createFlywayForTenant(AbstractTenantDatasourceProvider.getDefaultSchema(), schema));
        } catch (Exception e) {
            log.error("创建租户schema异常：", e);
            return ApiResult.fail(e.getMessage());
        }

        log.info("创建新租户schema：【{}】成功！", schema);
        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<Boolean> schemaDrop(String schema) {
        if (!TenantClient.enabledTenant()) {
            log.warn("未启用租户功能");
            return ApiResult.ok(true, "未启用租户功能");
        }
        if (tenantClientProperties.getIsolateStrategy() == TenantIsolateStrategy.NONE
                || tenantClientProperties.getIsolateStrategy() == TenantIsolateStrategy.FIELD) {
            log.warn("当前策略{}无需删除", tenantClientProperties.getIsolateStrategy());
            return ApiResult.ok(true, "当前隔离策略无需删除");
        }
        log.info("删除租户schema：{}", schema);

        Database database = AbstractTenantDatasourceProvider.getDatabaseType();
        if (database == null) {
            return ApiResult.fail("未知数据库类型");
        }

        // 开始删除schema
        boolean result = dropSchema(database, normalizeSchemaName(schema));
        return ApiResult.ok(result);
    }

    private String normalizeSchemaName(String schema) {
        String prefix = AbstractTenantDatasourceProvider.getDefaultSchema();
        prefix = CharSequenceUtil.isBlank(prefix) ? "" : prefix + "_";

        return prefix + schema;
    }

    private boolean dropSchema(Database database, String schema) {
        try {
            String sql = SqlUtil.generateExistsSchemaSql(database, schema);
            boolean exists = !jdbcTemplate.queryForList(sql).isEmpty();
            if (!exists) {
                log.info("删除schema失败，schema 【{}】不存在", schema);
                return true;
            }

            sql = SqlUtil.generateDropSchemaSql(database, schema);
            int result = jdbcTemplate.update(sql);
            log.info("删除schema【{}】结果：{}", schema, result);

            return true;
        } catch (Exception e) {
            log.error("删除schema失败：", e);
        }

        return false;
    }

}
