package com.elitescloud.boot.tenant.client.support.database.mysql;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.db.meta.Column;
import cn.hutool.db.meta.ColumnIndexInfo;
import cn.hutool.db.meta.IndexInfo;
import cn.hutool.db.meta.Table;
import com.elitescloud.boot.tenant.client.common.DataBaseExport;
import com.elitescloud.boot.tenant.client.support.database.AbstractExporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.sql.Types;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * MySQL导出工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/27
 */
public class MySqlExporter extends AbstractExporter {
    private static final Logger LOGGER = LoggerFactory.getLogger(MySqlExporter.class);

    public MySqlExporter(DataBaseExport dataBaseExport) {
        super(dataBaseExport);
    }

    @Override
    protected void beforeExport() {
        // nothing to do
    }

    @Override
    protected String generateTableDdl(Table table) {
        StringBuilder result = new StringBuilder();
        result.append(generateTable(table));
        result.append(" ( \n");

        // 列信息
        String columns = generateColumns(table);
        result.append(columns);

        // 索引信息
        String keys = generateKeys(table);
        result.append(StringUtils.hasText(keys) ? ", \n" + keys + "\n" : "\n");

        result.append(String.format(") COMMENT = '%s'; ", CharSequenceUtil.blankToDefault(table.getComment(), table.getTableName())));

        return result.toString();
    }

    private String generateTable(Table table) {
        String tableName = sensitiveName(table.getTableName(), dataBaseExport.isTableNameSensitive());
        switch (dataBaseExport.getTableCreate()) {
            case CREATE:
                // 直接创建
                return String.format("CREATE TABLE %s ", tableName);
            case DROP_BEFORE_CREATE:
                // 创建前先删除
                return String.format("DROP TABLE IF EXISTS %s;%n" +
                        "CREATE TABLE %s ", tableName, tableName);
            case CREATE_IF_NO_EXISTS:
                // 如果不存在则创建
                return String.format("CREATE TABLE IF NOT EXISTS %s ", tableName);
            default:
                throw new RuntimeException("未知创建策略");
        }
    }

    private String sensitiveName(String name, boolean sensitive) {
        return sensitive ? name : "`" + name + "`";
    }

    private String generateColumns(Table table) {
        List<String> ddlList = new ArrayList<>();

        for (Column column : table.getColumns()) {
            StringBuilder result = new StringBuilder();
            // 列名称
            result.append(String.format("%s    ", sensitiveName(column.getName(), dataBaseExport.isColumnNameSensitive())));

            // 列类型
            result.append(normalizeColumnType(column));

            // 是否为空
            result.append(normalizeNullable(column));

            // 是否自增
            if (column.isAutoIncrement()) {
                result.append("AUTO_INCREMENT ");
            }

            // 字段注释
            result.append(String.format("COMMENT '%s'", normalizeComment(CharSequenceUtil.blankToDefault(column.getComment(), column.getName()))));

            ddlList.add(result.toString());
        }

        return String.join(", \n", ddlList);
    }

    private String normalizeColumnType(Column column) {
        Set<Integer> noSizeTypes = Set.of(Types.DATE, Types.TIME, Types.TIMESTAMP, Types.TIME_WITH_TIMEZONE, Types.TIMESTAMP_WITH_TIMEZONE,
                Types.LONGVARBINARY, Types.BLOB, Types.CLOB, Types.NCLOB, Types.LONGVARCHAR,
                Types.DOUBLE);

        String typeName = column.getTypeName().toLowerCase();
        if (noSizeTypes.contains(column.getType())) {
            return String.format("%s    ", typeName);
        }

        if (typeName.contains("unsigned")) {
            // 数字类型
            return String.format("%s(%s) unsigned    ", typeName.split(" ")[0], column.getSize());
        }

        return String.format("%s(%s)    ", column.getTypeName().toLowerCase(), column.getSize());
    }

    private String normalizeNullable(Column column) {
        if (column.isNullable()) {
            // 非字符串的
            Set<Integer> originalTypeValue = Set.of(Types.TIMESTAMP, Types.DATE, Types.TIME, Types.TIME_WITH_TIMEZONE, Types.TIMESTAMP_WITH_TIMEZONE);
            if (StringUtils.hasText(column.getColumnDef())) {
                if (originalTypeValue.contains(column.getType())) {
                    return String.format("DEFAULT %s ", column.getColumnDef());
                }
                return String.format("DEFAULT '%s' ", column.getColumnDef());
            }

            if (column.getType() == Types.TIMESTAMP) {
                return "NULL DEFAULT NULL ";
            }
            return "DEFAULT NULL ";
        }
        return "NOT NULL ";
    }

    private String normalizeComment(String comment) {
        if (!StringUtils.hasText(comment)) {
            return null;
        }

        while (comment.startsWith("'")) {
            comment = comment.substring(1);
        }
        while (comment.endsWith("'")) {
            comment = comment.substring(0, comment.length() - 1);
        }

        return comment;
    }

    private String generateKeys(Table table) {
        List<IndexInfo> indexInfoList = table.getIndexInfoList();
        if (CollectionUtils.isEmpty(indexInfoList)) {
            return null;
        }

        List<String> ddlList = new ArrayList<>();
        for (IndexInfo indexInfo : indexInfoList) {
            if (indexInfo.getIndexName().equals("PRIMARY")) {
                // 主键
                ddlList.add(String.format("PRIMARY KEY (%s)", generateKeyColumns(indexInfo.getColumnIndexInfoList())));
                continue;
            }

            ddlList.add((indexInfo.isNonUnique() ? "KEY " : "UNIQUE KEY ") + String.format("`%s` (%s)", indexInfo.getIndexName(), generateKeyColumns(indexInfo.getColumnIndexInfoList())));
        }
        return String.join(", \n", ddlList);
    }

    private String generateKeyColumns(List<ColumnIndexInfo> indexInfoList) {
        return indexInfoList.stream().map(t -> String.format("`%s`", t.getColumnName())).collect(Collectors.joining(", "));
    }
}
