package com.elitescloud.boot.tenant.client.common;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 租户客户端工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/6
 */
@Log4j2
public class TenantClient {

    private static TenantProvider tenantProvider;
    private static final AtomicBoolean INITED = new AtomicBoolean(false);

    protected TenantClient() {
    }

    /**
     * 是否启用了租户
     *
     * @return 是否启用了租户
     */
    public static boolean enabledTenant() {
        return TenantContextHolder.isEnabledTenant();
    }

    /**
     * 获取当前租户信息
     *
     * @return 租户信息
     */
    public static SysTenantDTO getCurrentTenant() {
        if (!enabledTenant()) {
            // 未启用租户
            return null;
        }
        return obtainTenant();
    }

    /**
     * 获取当前Session中的租户
     * <p>
     * 当上下文中的租户不满足时，通过数据隔离工具进行指定的租户
     *
     * @return
     */
    public static SysTenantDTO getSessionTenant() {
        if (TenantSession.getUseDefault() || TenantSession.getNoTenant()) {
            // 指定使用默认的
            return null;
        }

        return ObjectUtil.defaultIfNull(TenantSession.getCurrentTenant(), getCurrentTenant());
    }

    /**
     * 获取租户信息
     *
     * @param tenantId 租户ID
     * @return 租户信息
     */
    public static SysTenantDTO getTenant(Long tenantId) {
        if (!enabledTenant()) {
            // 未启用租户
            return null;
        }
        if (tenantId == null) {
            return null;
        }
        return getTenantProvider().getById(tenantId).get();
    }

    /**
     * 获取租户信息
     *
     * @param code 租户编码
     * @return 租户信息
     */
    public static SysTenantDTO getTenant(String code) {
        if (!enabledTenant()) {
            // 未启用租户
            return null;
        }
        if (code == null) {
            return null;
        }
        return getTenantProvider().getByCode(code).get();
    }

    /**
     * 获取所有的租户信息
     *
     * @return 租户信息
     */
    public static List<SysTenantDTO> getAllTenants() {
        if (!enabledTenant()) {
            // 未启用租户
            return Collections.emptyList();
        }
        return new ArrayList<>(getTenantProvider().getAllTenant().values());
    }

    /**
     * 从当前请求中获取租户
     *
     * @return 租户
     */
    public static SysTenantDTO obtainTenantFromRequest() {
        if (!enabledTenant()) {
            // 未启用租户
            return null;
        }
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            return null;
        }
        return TenantRequestUtil.obtainTenant(((ServletRequestAttributes) requestAttributes).getRequest());
    }

    /**
     * 清理缓存
     */
    public static void clearCache() {
        if (!enabledTenant()) {
            // 未启用租户
            return;
        }

        getTenantProvider().clearCache();
    }

    private static TenantProvider getTenantProvider() {
        if (INITED.compareAndSet(false, true)) {
            tenantProvider = SpringContextHolder.getObjectProvider(TenantProvider.class).getIfAvailable();
        }
        return tenantProvider;
    }

    private static SysTenantDTO obtainTenant() {
        // 然后从租户上下文中获取
        SysTenantDTO tenant = TenantContextHolder.getCurrentTenant();
        if (tenant != null) {
            return tenant;
        }

        // 从当前用户获取
        var currentUser = SecurityContextUtil.currentUser();
        if (currentUser != null) {
            tenant = currentUser.getTenant();
            if (tenant != null) {
                return tenant;
            }
        }

        // 最后从请求信息中获取
        var requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes != null) {
            return TenantRequestUtil.obtainTenant(((ServletRequestAttributes) requestAttributes).getRequest());
        }

        return null;
    }
}
