package com.elitescloud.boot.tenant.client.support.config;

import com.elitescloud.boot.provider.TenantOrgDataIsolateProvider;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;
import org.springframework.util.Assert;

/**
 * 租户组织数据拦截.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/4
 */
@Aspect
public class TenantOrgTransactionAspect {

    private final TenantOrgDataIsolateProvider tenantOrgDataIsolateProvider;

    public TenantOrgTransactionAspect(TenantOrgDataIsolateProvider tenantOrgDataIsolateProvider) {
        this.tenantOrgDataIsolateProvider = tenantOrgDataIsolateProvider;
    }

    @Pointcut("@annotation(com.elitescloud.cloudt.core.annotation.TenantOrgTransaction)")
    private void pointCutMethod() {
    }

    @Pointcut("@within(com.elitescloud.cloudt.core.annotation.TenantOrgTransaction)")
    private void pointCutClass() {
    }

    @Around("pointCutClass() || pointCutMethod()")
    @SuppressWarnings("unchecked")
    public Object cutAround(ProceedingJoinPoint point) throws Throwable {
        // 先获取注解
        TenantOrgTransaction annotation = obtainAnnotation(point);
        Assert.notNull(annotation, "未获取到TenantOrgTransaction注解信息");

        // 是否使用租户组织
        var isUseTenantOrg = annotation.useTenantOrg();
        if (isUseTenantOrg) {
            return tenantOrgDataIsolateProvider.byCurrentTenantOrg(() -> {
                try {
                    return point.proceed();
                } catch (Throwable e) {
                    throw ExceptionsUtil.wrapRuntime(e);
                }
            });
        }

        return tenantOrgDataIsolateProvider.byNoTenantOrg(() -> {
            try {
                return point.proceed();
            } catch (Throwable e) {
                throw ExceptionsUtil.wrapRuntime(e);
            }
        });
    }

    private TenantOrgTransaction obtainAnnotation(ProceedingJoinPoint point) {
        if (point instanceof MethodInvocationProceedingJoinPoint) {
            // 优先获取方法上的注解
            TenantOrgTransaction annotation = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(TenantOrgTransaction.class);

            if (annotation == null) {
                annotation = point.getThis().getClass().getAnnotation(TenantOrgTransaction.class);
            }

            return annotation;
        }
        return null;
    }
}
