package com.elitescloud.boot.tenant.client.common;

import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;

/**
 * 数据源提供者的父类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/25
 */
@Log4j2
public abstract class AbstractTenantDatasourceProvider {

    /**
     * 默认schema
     */
    private static String defaultSchema;

    /**
     * 默认数据源
     */
    private static DataSource defaultDataSource;

    /**
     * 数据库类型
     */
    private static Database databaseType;

    /**
     * 数据隔离策略
     */
    private static TenantIsolateStrategy isolateStrategy = TenantIsolateStrategy.NONE;

    private AbstractTenantDatasourceProvider() {
    }

    /**
     * 分析当前适合的租户
     *
     * @return 租户
     */
    public static SysTenantDTO detectedTenant() {
        // 从当前上下文获取当前租户
        return TenantClient.getSessionTenant();
    }

    /**
     * 获取默认schema
     *
     * @return 默认schema
     */
    public static String getDefaultSchema() {
        return defaultSchema;
    }

    /**
     * 设置默认schemaName
     *
     * @param defaultSchema 默认schema名称
     */
    public static void setDefaultSchema(String defaultSchema) {
        if (StringUtils.hasText(AbstractTenantDatasourceProvider.defaultSchema)
                || !StringUtils.hasText(defaultSchema)) {
            return;
        }

        AbstractTenantDatasourceProvider.defaultSchema = defaultSchema;
    }

    /**
     * 获取数据源
     *
     * @return 数据源
     */
    public static DataSource getDefaultDataSource() {
        return defaultDataSource;
    }

    /**
     * 设置默认数据源
     *
     * @param defaultDataSource 数据源
     */
    public static void setDefaultDataSource(DataSource defaultDataSource) {
        if (defaultDataSource == null) {
            return;
        }
        AbstractTenantDatasourceProvider.defaultDataSource = defaultDataSource;
    }

    /**
     * 获取数据库类型
     *
     * @return 数据库类型
     */
    public static Database getDatabaseType() {
        return databaseType;
    }

    /**
     * 设置默认数据库类型
     *
     * @param databaseType 数据库类型
     */
    public static void setDatabaseType(Database databaseType) {
        if (databaseType == null || AbstractTenantDatasourceProvider.databaseType != null) {
            return;
        }
        AbstractTenantDatasourceProvider.databaseType = databaseType;
    }

    /**
     * 根据数据库连接解析数据库类型
     *
     * @param url 数据库连接
     * @return 数据库类型
     */
    public static Database getDatabaseTypeByUrl(String url) {
        if (url == null) {
            return null;
        }
        int index = url.indexOf("://");

        String temp;
        if (index > 0) {
            temp = url.substring(0, index);
        } else {
            return null;
        }

        index = temp.lastIndexOf(":");
        if (index > 0) {
            temp = temp.substring(index + 1);
        }
        try {
            return Database.valueOf(temp.toUpperCase());
        } catch (IllegalArgumentException e) {
            log.error("解析数据库类型失败：{}", url);
            return null;
        }
    }

    /**
     * 设置数据隔离策略
     *
     * @param isolateStrategy 数据隔离策略
     */
    public static void setIsolateStrategy(TenantIsolateStrategy isolateStrategy) {
        if (isolateStrategy == null) {
            return;
        }
        log.debug("租户隔离策略：{}", isolateStrategy);
        AbstractTenantDatasourceProvider.isolateStrategy = isolateStrategy;
    }

    /**
     * 获取数据隔离策略
     *
     * @return 是否数据分离
     */
    public static TenantIsolateStrategy getIsolateStrategy() {
        return isolateStrategy;
    }
}
