package com.elitescloud.boot.tenant.client.support.config;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.wrapper.Isolatable;
import com.elitescloud.boot.wrapper.TransactionWrapper;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;
import org.springframework.core.Ordered;
import org.springframework.util.Assert;

/**
 * 租户事务处理拦截.
 * <p>
 * 依据当前用户信息结合注解处理schema的切换
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/7
 */
@Log4j2
@Aspect
public class TenantTransactionAspect implements Ordered {

    private final TransactionWrapper tenantTransactionWrapper;

    public TenantTransactionAspect(TransactionWrapper tenantTransactionWrapper) {
        this.tenantTransactionWrapper = tenantTransactionWrapper;
    }

    @Pointcut("@annotation(com.elitescloud.cloudt.core.annotation.TenantTransaction)")
    private void pointCutMethod() {
    }

    @Pointcut("@within(com.elitescloud.cloudt.core.annotation.TenantTransaction)")
    private void pointCutClass() {
    }

    @Around("pointCutClass() || pointCutMethod()")
    public Object cutAround(ProceedingJoinPoint point) throws Throwable {
        // 先获取注解
        TenantTransaction annotation = obtainAnnotation(point);
        Assert.notNull(annotation, "未获取到TenantTransaction注解信息");

        // 当前用户
        GeneralUserDetails currentUser = SecurityContextUtil.currentUser();
        if (annotation.supportOperation() && currentUser != null && currentUser.isOperation()) {
            // 不隔离
            return execute(point, Isolatable.NONE);
        }

        // 获取隔离的租户
        var tenant = obtainTenant(annotation, currentUser);
        // 执行业务
        return execute(point, tenant);
    }

    @SuppressWarnings("unchecked")
    private Object execute(ProceedingJoinPoint point, Object param) throws Throwable {
        return tenantTransactionWrapper.apply(() -> {
            try {
                return point.proceed();
            } catch (Throwable e) {
                ExceptionsUtil.wrapAndThrow(e);
            }
            return null;
        }, param);
    }

    private TenantTransaction obtainAnnotation(ProceedingJoinPoint point) {
        if (point instanceof MethodInvocationProceedingJoinPoint) {
            // 优先获取方法上的注解
            TenantTransaction annotation = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(TenantTransaction.class);

            if (annotation == null) {
                annotation = point.getThis().getClass().getAnnotation(TenantTransaction.class);
            }

            return annotation;
        }
        return null;
    }

    private SysTenantDTO obtainTenant(TenantTransaction annotation, GeneralUserDetails currentUser) {
        TenantIsolateType isolateType = annotation.isolateType();

        // 默认的优先级最高
        if (isolateType == TenantIsolateType.DEFAULT) {
            return null;
        }
        if (TenantSession.getUseDefault() || TenantSession.getNoTenant()) {
            return null;
        }

        // 获取租户
        SysTenantDTO currentTenant = ObjectUtil.defaultIfNull(TenantSession.getCurrentTenant(), TenantContextHolder.getCurrentTenant());
        if (currentTenant != null) {
            // 租户的
            if (isolateType == TenantIsolateType.TENANT || currentUser == null) {
                return currentTenant;
            }

            // 仅租户用户可访问租户的
            if (isolateType == TenantIsolateType.TENANT_USER) {
                if (currentUser.isSystemAdmin() || currentUser.isOperation() || currentUser.isTenantAdmin()) {
                    return null;
                }
                return currentTenant;
            }

            return null;
        }

        if (annotation.tenantRequired()) {
            throw new BusinessException("未获取到当前租户信息");
        }

        return null;
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }
}
