package com.elitescloud.boot.tenant.client;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.condition.ConditionalOnRpc;
import com.elitescloud.boot.condition.ConditionalOnTenant;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.flyway.common.FlywayBuilder;
import com.elitescloud.boot.flyway.common.FlywayHelper;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.tenant.client.common.TenantDatabaseRpcProvider;
import com.elitescloud.boot.tenant.client.support.TenantProviderService;
import com.elitescloud.boot.tenant.client.support.config.DefaultTenantClientProvider;
import com.elitescloud.boot.tenant.client.support.impl.*;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.tenant.provider.TenantProvider;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.config.spring.context.annotation.DubboComponentScan;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.task.TaskExecutor;

import java.util.Collections;

/**
 * 租户客户端自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/24
 */
@ConditionalOnTenant
@EnableConfigurationProperties(TenantClientProperties.class)
@Import({CloudtTenantClientAutoConfiguration.OnDubbo.class, CloudtTenantClientAutoConfiguration.OnOpenFeign.class})
@Log4j2
public class CloudtTenantClientAutoConfiguration {

    private final TenantClientProperties clientProperties;

    public CloudtTenantClientAutoConfiguration(TenantClientProperties clientProperties) {
        this.clientProperties = clientProperties;
        log.info("启用租户，租户隔离方式：{}", clientProperties.getIsolateStrategy());
        if (clientProperties.getIsolateStrategy() == TenantIsolateStrategy.SCHEMA) {
            log.info("默认schema：{} ", clientProperties.getDefaultSchema());
            Assert.notBlank(clientProperties.getDefaultSchema(), "租户使用Schema隔离时默认schema不能为空");
        }
    }

    /**
     * 租户服务提供者
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    public TenantProviderService tenantProviderService() {
        log.warn("未发现有效的租户服务提供者");
        return () -> ApiResult.ok(Collections.emptyList());
    }

    @Bean
    public TenantProvider defaultTenantProvider(TenantProviderService tenantProviderService,
                                                ObjectProvider<RedisUtils> redisUtils,
                                                RedisWrapper redisWrapper) {
        return new DefaultTenantProvider(tenantProviderService, redisUtils.getIfAvailable(), redisWrapper);
    }

    /**
     * 客户端同步provider
     *
     * @param tenantProvider
     * @param flywayHelper
     * @param flywayBuilder
     * @return
     */
    @Bean
    @ConditionalOnTenant(isolateStrategy = {TenantIsolateStrategy.SCHEMA, TenantIsolateStrategy.DATABASE})
    public ClientSyncTenantProvider clientSyncTenantProvider(TenantProvider tenantProvider,
                                                             FlywayHelper flywayHelper,
                                                             FlywayBuilder flywayBuilder) {
        return new ClientSyncTenantProvider(clientProperties, tenantProvider, flywayHelper, flywayBuilder);
    }

    @Bean
    @ConditionalOnTenant(isolateStrategy = {TenantIsolateStrategy.SCHEMA, TenantIsolateStrategy.DATABASE})
    public TenantSchemaInitializer tenantSchemaInitializer(ClientSyncTenantProvider clientSyncTenantProvider, TaskExecutor taskExecutor) {
        return new TenantSchemaInitializer(clientSyncTenantProvider, taskExecutor, clientProperties);
    }

    @Bean
    public TenantRequestInterceptor tenantRequestInterceptor() {
        return new TenantRequestInterceptor(clientProperties);
    }

    @Bean
    public TenantClientProvider tenantClientProvider() {
        return new DefaultTenantClientProvider(clientProperties);
    }

    @Bean
    public TenantContextTransfer tenantContextTransfer(com.elitescloud.boot.provider.TenantClientProvider tenantClientProvider) {
        return new TenantContextTransfer(tenantClientProvider);
    }

    @Bean
    public TenantOrgContextTransfer tenantOrgContextTransfer() {
        return new TenantOrgContextTransfer();
    }

    @ConditionalOnRpc(requiredDubbo = true)
    @DubboComponentScan(basePackages = "com.elitesland.cloudt.tenant.rpc.consumer")
    static class OnDubbo {

        public OnDubbo() {
            log.debug("租户客户端已启用dubbo远程调用");
        }
    }

    @ConditionalOnRpc(requiredOpenFeign = true)
    static class OnOpenFeign {

        public OnOpenFeign() {
            log.debug("租户客户端已启用OpenFeign远程调用");
        }

        @Bean
        public TenantDatabaseRpcProvider tenantDatabaseRpcProvider() {
            return new TenantDatabaseRpcProviderImpl();
        }
    }
}
