package com.elitescloud.boot.tenant.client.support.database.oracle;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.db.meta.Column;
import cn.hutool.db.meta.ColumnIndexInfo;
import cn.hutool.db.meta.IndexInfo;
import cn.hutool.db.meta.Table;
import com.elitescloud.boot.tenant.client.common.DataBaseExport;
import com.elitescloud.boot.tenant.client.support.database.AbstractExporter;
import com.elitescloud.boot.tenant.client.support.database.TableCreate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.sql.Types;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 通用的导出工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/30
 */
public class OracleExporter extends AbstractExporter {
    private static final Logger LOGGER = LoggerFactory.getLogger(OracleExporter.class);

    /**
     * 索引名称不能重复
     */
    private Map<String, String> existsIndexNames = new HashMap<>();
    private Map<String, String> existsPrimaryNames = new HashMap<>();
    /**
     * 索引的最大长度
     */
    private int maxIndexLength = 30;

    public OracleExporter(DataBaseExport dataBaseExport) {
        super(dataBaseExport);
    }

    @Override
    protected void beforeExport() {

    }

    @Override
    protected String generateTableDdl(Table table) {
        String tableName = table.getTableName();

        switch (dataBaseExport.getTableCreate()) {
            case CREATE:
                // 直接创建
                return generateTableDirectly(table);
            case DROP_BEFORE_CREATE:
                // 创建前先删除
                return String.format("DECLARE\n" +
                        "  table_num int(8);\n" +
                        "  BEGIN\n" +
                        "    SELECT COUNT(*) INTO table_num FROM all_tables WHERE table_name = '%s' AND owner = (SELECT sys_context('userenv', 'current_schema' ) FROM dual);\n" +
                        "    IF(table_num = 1) THEN\n" +
                        "       EXECUTE IMMEDIATE 'DROP TABLE \"%s\"';\n" +
                        "    END IF;\n" +
                        "%s" +
                        "  END;", tableName, tableName, generateTableDynamic(table));
            case CREATE_IF_NO_EXISTS:
                // 如果不存在则创建
                return String.format("DECLARE\n" +
                        "  table_num int(8);\n" +
                        "  BEGIN\n" +
                        "    SELECT COUNT(*) INTO table_num FROM all_tables WHERE table_name = '%s' AND owner = (SELECT sys_context('userenv', 'current_schema' ) FROM dual);\n" +
                        "    IF(table_num = 0) THEN\n" +
                        "%s" +
                        "    END IF;\n" +
                        "  END;", tableName, generateTableDynamic(table));
            default:
                throw new RuntimeException("未知创建策略");
        }
    }

    private String generateTableDirectly(Table table) {
        String tableName = sensitiveName(table.getTableName(), dataBaseExport.isTableNameSensitive());
        StringBuilder result = new StringBuilder();
        result.append(String.format("CREATE TABLE %s ( %n", tableName));

        // 列信息
        String columns = generateColumns(table);
        result.append(columns);

        result.append("); \n");

        // 表注释信息
        result.append(String.format("COMMENT ON TABLE %s IS '%s'; %n", tableName, CharSequenceUtil.blankToDefault(table.getComment(), table.getTableName())));
        // 列注释
        result.append(generateColumnComment(table));

        // 索引信息
        result.append(generateKeys(table));

        return result.toString();
    }

    private String generateTableDynamic(Table table) {
        String tableName = sensitiveName(table.getTableName(), dataBaseExport.isTableNameSensitive());
        String columnDdl = generateColumns(table);
        StringBuilder result = new StringBuilder();

        // 创建表
        result.append(String.format("\tEXECUTE IMMEDIATE 'CREATE TABLE %s ( ' ||\n" +
                        "%s" +
                        "\t\t\t\t\t\t')';\n"
                , tableName, columnDdl));
        // 表注释
        result.append("\tEXECUTE IMMEDIATE '")
                .append(String.format("COMMENT ON TABLE %s IS ''%s'' ", tableName, CharSequenceUtil.blankToDefault(table.getComment(), table.getTableName())))
                .append("'; \n");
        // 列注释
        result.append(generateColumnComment(table));

        // 索引信息
        result.append(generateKeys(table));

        return result.toString();
    }

    private String sensitiveName(String name, boolean sensitive) {
        return sensitive ? name : "\"" + name + "\"";
    }

    private String generateColumns(Table table) {
        List<String> ddlList = new ArrayList<>();

        for (Column column : table.getColumns()) {
            StringBuilder result = new StringBuilder();
            // 列名称
            result.append(String.format("%s    ", sensitiveName(column.getName(), dataBaseExport.isColumnNameSensitive())));

            // 列类型
            result.append(normalizeColumnType(column).toUpperCase());

            // 是否为空
            result.append(normalizeNullable(column));

            // 是否自增
            if (column.isAutoIncrement()) {
                // oracle没有自增
            }

            ddlList.add(result.toString());
        }

        if (!dynamic()) {
            return String.join(", \n", ddlList) + "\n";
        }
        return "\t\t\t\t\t\t\t'" + String.join(",' || \n \t\t\t\t\t\t\t'", ddlList) + "' || \n";
    }

    private String generateColumnComment(Table table) {
        String tableName = sensitiveName(table.getTableName(), dataBaseExport.isTableNameSensitive());
        StringBuilder result = new StringBuilder();

        boolean dynamic = dynamic();
        for (Column column : table.getColumns()) {
            // 注释内容
            String comment = normalizeComment(CharSequenceUtil.blankToDefault(column.getComment(), column.getName()));

            if (dynamic) {
                result.append("\tEXECUTE IMMEDIATE '")
                        .append(String.format("COMMENT ON COLUMN %s.%s IS ''%s'' ", tableName, sensitiveName(column.getName(), dataBaseExport.isColumnNameSensitive()), comment)).append("';\n");
                continue;
            }
            result.append(String.format("COMMENT ON COLUMN %s.%s IS '%s';", tableName, sensitiveName(column.getName(), dataBaseExport.isColumnNameSensitive()), comment)).append("\n");
        }

        return result.toString();
    }

    private String normalizeColumnType(Column column) {
        switch (column.getType()) {
            case Types.BIT:
                return "number(1, 0) ";
            case Types.TINYINT:
                return "number(3, 0) ";
            case Types.SMALLINT:
                return "number(5, 0) ";
            case Types.INTEGER:
                return "number(10, 0) ";
            case Types.BIGINT:
                return "number(19, 0) ";
            case Types.FLOAT:
                return String.format("float(%s, %s) ", column.getSize(), column.getDigit());
            case Types.REAL:
                return "float(24) ";
            case Types.DOUBLE:
            case Types.NUMERIC:
            case Types.DECIMAL:
                return String.format("number(%s, %s) ", column.getSize(), column.getDigit());
            case Types.CHAR:
            case Types.NCHAR:
                return String.format("char(%s) ", column.getSize());
            case Types.VARCHAR:
            case Types.NVARCHAR:
                if (column.getSize() > 2000) {
                    // nvarchar2最大2000
                    return "clob ";
                }
                return String.format("nvarchar2(%s) ", column.getSize());
            case Types.LONGVARCHAR:
            case Types.CLOB:
            case Types.NCLOB:
                return "clob ";
            case Types.DATE:
            case Types.TIME:
            case Types.TIMESTAMP:
                return "date ";
            case Types.BINARY:
            case Types.VARBINARY:
                return "raw ";
            case Types.LONGVARBINARY:
            case Types.BLOB:
                return "blob ";
            default:
                return "nvarchar2 ";
        }
    }

    private String normalizeNullable(Column column) {
        if (column.isNullable()) {
            // 非字符串的
            Set<Integer> dateTypeValue = Set.of(Types.TIMESTAMP, Types.DATE, Types.TIME, Types.TIME_WITH_TIMEZONE, Types.TIMESTAMP_WITH_TIMEZONE);
            if (StringUtils.hasText(column.getColumnDef())) {
                if (dateTypeValue.contains(column.getType())) {
                    return "DEFAULT sysdate ";
                }
                return String.format(dynamic() ? "DEFAULT ''%s'' " : "DEFAULT '%s' ", column.getColumnDef());
            }

            return "";
        }
        return "NOT NULL ";
    }

    private String normalizeComment(String comment) {
        if (!StringUtils.hasText(comment)) {
            return null;
        }

        while (comment.startsWith("'")) {
            comment = comment.substring(1);
        }
        while (comment.endsWith("'")) {
            comment = comment.substring(0, comment.length() - 1);
        }

        return comment;
    }

    private String generateKeys(Table table) {
        List<IndexInfo> indexInfoList = table.getIndexInfoList();
        if (CollectionUtils.isEmpty(indexInfoList)) {
            return "";
        }

        // 索引排重
        indexInfoList = distinctIndexInfo(table);
        boolean dynamic = dynamic();

        StringBuilder result = new StringBuilder();
        for (IndexInfo indexInfo : indexInfoList) {
            if (indexInfo.getIndexName().equals(PRIMARY_NAME)) {
                // 主键
                String indexName = generateIndexName(indexInfo, true);

                if (dynamic) {
                    result.append("\tEXECUTE IMMEDIATE '")
                            .append(String.format("ALTER TABLE \"%s\" ADD CONSTRAINT \"%s\" PRIMARY KEY (%s)", table.getTableName(),
                                    indexName, generateKeyColumnNames(indexInfo.getColumnIndexInfoList())))
                            .append(" '; \n");
                    continue;
                }
                result.append(String.format("ALTER TABLE \"%s\" ADD CONSTRAINT \"%s\" PRIMARY KEY (%s);%n", table.getTableName(),
                        indexName, generateKeyColumnNames(indexInfo.getColumnIndexInfoList())));

                continue;
            }

            String indexName = generateIndexName(indexInfo, false);
            if (dynamic) {
                result.append("\tEXECUTE IMMEDIATE '")
                        .append(indexInfo.isNonUnique() ? "CREATE INDEX " : "CREATE UNIQUE INDEX ")
                        .append(String.format("\"%s\" on \"%s\" (%s)", indexName, table.getTableName(),
                                generateKeyColumns(indexInfo.getColumnIndexInfoList())))
                        .append(" '; \n");
                continue;
            }
            result.append(indexInfo.isNonUnique() ? "CREATE INDEX " : "CREATE UNIQUE INDEX ")
                    .append(String.format("\"%s\" on \"%s\" (%s);%n", indexName, table.getTableName(),
                            generateKeyColumns(indexInfo.getColumnIndexInfoList())));
        }
        return result.toString();
    }

    private String generateIndexName(IndexInfo indexInfo, boolean primaryKey) {
        Map<String, String> indexMapCache = primaryKey ? existsPrimaryNames : existsIndexNames;

        // 判断是否超出长度
        String indexName = primaryKey ? "pk_" + indexInfo.getTableName() : indexInfo.getIndexName();
        if (indexName.length() > maxIndexLength) {
            indexName = indexName.substring(0, maxIndexLength);
        }

        String existsIndexTableName = indexMapCache.get(indexName);
        if (!StringUtils.hasText(existsIndexTableName)) {
            // 不存在，直接使用
            indexMapCache.put(indexName, indexInfo.getTableName());
            return indexName;
        }

        // 存在则重新生成
        indexName = normalizeIndexName(indexInfo);
        if (indexName.length() <= maxIndexLength) {
            return indexName;
        }
        indexName = indexName.substring(0, maxIndexLength);

        int index = 1;
        int value = 0;
        indexName = indexName.substring(0, indexName.length() - index);
        String tempIndexName = indexName + value;
        while (indexMapCache.containsKey(tempIndexName)) {
            value++;
            tempIndexName = indexName + value;

            if ((value + "").endsWith("9")) {
                index++;
                indexName = indexName.substring(0, indexName.length() - index);
            }
        }

        indexMapCache.put(tempIndexName, indexInfo.getTableName());
        return tempIndexName;
    }

    private String generateKeyColumnNames(List<ColumnIndexInfo> indexInfoList) {
        return indexInfoList.stream().map(t -> String.format("\"%s\"", t.getColumnName())).collect(Collectors.joining(", "));
    }

    private String generateKeyColumns(List<ColumnIndexInfo> indexInfoList) {
        return indexInfoList.stream()
                .map(t -> String.format("\"%s\" %s", t.getColumnName(), CharSequenceUtil.equals(t.getAscOrDesc(), "A") ? "ASC" : "DESC"))
                .collect(Collectors.joining(", "));
    }

    private boolean dynamic() {
        return dataBaseExport.getTableCreate() != TableCreate.CREATE;
    }
}
