package com.elitescloud.boot.tenant.client.support.database;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.db.meta.ColumnIndexInfo;
import cn.hutool.db.meta.IndexInfo;
import cn.hutool.db.meta.MetaUtil;
import cn.hutool.db.meta.Table;
import com.elitescloud.boot.tenant.client.common.DataBaseExport;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/27
 */
public abstract class AbstractExporter {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractExporter.class);
    /**
     * 主键名称
     */
    protected static final String PRIMARY_NAME = "PRIMARY";

    protected final DataBaseExport dataBaseExport;

    protected AbstractExporter(DataBaseExport dataBaseExport) {
        this.dataBaseExport = dataBaseExport;
    }

    public void export(Consumer<String> tableDdlConsumer) {
        // 检查导出参数
        checkParams();
        // 导出前的操作
        beforeExport();

        // 构建数据源
        DataSource dataSource = buildDataSource();

        // 查询元数据
        List<Table> metadataList = null;
        try {
            metadataList = queryMetadata(dataSource);
        } catch (Exception e) {
            throw new RuntimeException("查询数据库表结构元数据失败：", e);
        }
        if (CollectionUtils.isEmpty(metadataList)) {
            LOGGER.warn("未查询到有效的表结构");
            return;
        }

        // 导出到inputStream
        write(metadataList, tableDdlConsumer);
    }

    /**
     * 开始执行导出前的操作
     */
    protected abstract void beforeExport();

    /**
     * 生成表的ddl
     *
     * @param table 表元数据
     * @return ddl
     */
    protected abstract String generateTableDdl(Table table);

    /**
     * 索引排重
     *
     * @param table 表
     * @return 排重后的索引信息
     */
    protected List<IndexInfo> distinctIndexInfo(Table table) {
        return table.getIndexInfoList().stream().collect(Collectors.groupingBy(t -> t.getColumnIndexInfoList().stream().map(ColumnIndexInfo::getColumnName).collect(Collectors.joining("_")),
                Collectors.collectingAndThen(Collectors.toList(), tt -> {
                    var uniqueKey = tt.stream().filter(ttt -> !ttt.isNonUnique()).findAny();
                    // 优先取唯一性key
                    return uniqueKey.orElseGet(() -> tt.get(0));

                }))).values().stream().collect(Collectors.toList());
    }

    /**
     * 缩写名称
     *
     * @param name 名称
     * @return 缩写
     */
    private String shortForName(String name) {
        return Arrays.stream(name.split("_")).map(t -> {
            if (!StringUtils.hasText(t)) {
                return t;
            }
            return t.charAt(0) + t.substring(t.length() - 1);
        }).collect(Collectors.joining());
    }

    /**
     * 正常化索引名称
     *
     * @param indexInfo 索引信息
     * @return 索引名称
     */
    protected String normalizeIndexName(IndexInfo indexInfo) {
        String prefix = CharSequenceUtil.equals(indexInfo.getIndexName().toUpperCase(), PRIMARY_NAME) ? "pk_" : (indexInfo.isNonUnique() ? "idx_" : "udx_");
        return prefix + indexInfo.getTableName() + "_" + indexInfo.getColumnIndexInfoList().stream().map(ColumnIndexInfo::getColumnName).collect(Collectors.joining("_"));
    }

    private void write(List<Table> metadataList, Consumer<String> tableDdlConsumer) {
        for (Table table : metadataList) {
            tableDdlConsumer.accept(generateTableDdl(table));
        }

        LOGGER.info("共{}表导出完毕！", metadataList.size());
    }

    private List<Table> queryMetadata(DataSource dataSource) {
        List<String> tables;
        if (ArrayUtil.isEmpty(dataBaseExport.getTables())) {
            // 获取所有表
            tables = MetaUtil.getTables(dataSource);
        } else {
            tables = Arrays.stream(dataBaseExport.getTables()).distinct().collect(Collectors.toList());
        }

        // 获取元数据
        if (tables.isEmpty()) {
            return Collections.emptyList();
        }
        return tables.stream()
                .map(t -> MetaUtil.getTableMeta(dataSource, t))
                .collect(Collectors.toList());
    }

    private DataSource buildDataSource() {
        HikariDataSource dataSource = new HikariDataSource();
        dataSource.setJdbcUrl(dataBaseExport.getJdbcUrl());
        dataSource.setUsername(dataBaseExport.getUsername());
        dataSource.setPassword(dataBaseExport.getPassword());
        dataSource.setDriverClassName(dataBaseExport.getDriverClassName());

        return dataSource;
    }

    private void checkParams() {
        Assert.hasText(dataBaseExport.getJdbcUrl(), "jdbcUrl为空");
        Assert.hasText(dataBaseExport.getDriverClassName(), "driverClassName为空");
        Assert.hasText(dataBaseExport.getUsername(), "username为空");
        Assert.hasText(dataBaseExport.getPassword(), "password为空");

        normalizeParam();
    }

    private void normalizeParam() {
        // 格式化url
        String url = dataBaseExport.getJdbcUrl();
        // 需要获取元数据
        if (!url.contains("useInformationSchema")) {
            url = url.contains("?") ? url + "&" : url + "?";
            dataBaseExport.setJdbcUrl(url + "useInformationSchema = true");
        }
    }
}
