package com.elitescloud.boot.tenant.client.support.impl;

import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.context.TenantOrgContext;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.cloudt.core.tenant.support.TenantOrgDataIsolateProvider;
import lombok.extern.log4j.Log4j2;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.validation.constraints.NotNull;
import java.util.function.Supplier;

/**
 * 租户组织数据隔离工具.
 * <p>
 * 配置不使用租户组织的优先级最高，其次使用传递的租户组织，最后使用当前用户的租户组织
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/4
 */
@Log4j2
public class DefaultTenantOrgDataIsolateProvider implements TenantOrgDataIsolateProvider {
    private static final String TRANSACTION_NAME_PREFIX = "cloudt_tenantOrg";

    private final PlatformTransactionManager transactionManager;

    public DefaultTenantOrgDataIsolateProvider(PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

    @Override
    public <T> T byNoTenantOrg(@NotNull Supplier<T> supplier) {
        Boolean used = TenantOrgContext.isUseTenantOrg();
        if (used != null && !used) {
            // 目前已是不使用
            return supplier.get();
        }

        TenantOrgContext.setUseTenantOrg(false);
        // 创建新事务
        String newTransactionName = generateTransactionName(null);
        log.debug("Transaction execute：{} -> {}", TransactionSynchronizationManager::getCurrentTransactionName, () -> newTransactionName);
        TransactionStatus transactionStatus = buildTenantTransactionDefinition(newTransactionName);
        try {
            T result = supplier.get();
            // 提交事务
            afterCompletion(transactionStatus, null);
            return result;
        } catch (Exception e) {
            // 回滚事务
            afterCompletion(transactionStatus, e);
            throw ExceptionsUtil.wrapRuntime(e);
        } finally {
            if (used == null) {
                TenantOrgContext.clearUseTenantOrg();
            } else {
                // 恢复之前的
                TenantOrgContext.setUseTenantOrg(true);
            }
        }
    }

    @Override
    public <T> T byCurrentTenantOrg(@NotNull Supplier<T> supplier) {
        var currentUser = SecurityContextUtil.currentUser();
        return execute(supplier, currentUser == null ? null : currentUser.getTenantOrgId());
    }

    @Override
    public <T> T byTenantOrg(@NotNull Supplier<T> supplier, @NotNull Long tenantOrgId) {
        return execute(supplier, tenantOrgId);
    }

    private <T> T execute(@NotNull Supplier<T> supplier, Long tenantOrgId) {
        Boolean used = TenantOrgContext.isUseTenantOrg();
        Long oldTenantOrgId = TenantOrgContext.getTenantOrg();

        // 如果当前不使用，则先取消
        if (used != null && !used) {
            TenantOrgContext.setUseTenantOrg(true);
        }

        // 设置要过滤的租户组织
        TenantOrgContext.setTenantOrgId(tenantOrgId);
        // 创建新事务
        String newTransactionName = generateTransactionName(null);
        log.debug("Transaction execute：{} -> {}", TransactionSynchronizationManager::getCurrentTransactionName, () -> newTransactionName);
        TransactionStatus transactionStatus = buildTenantTransactionDefinition(newTransactionName);
        try {
            T result = supplier.get();
            // 提交事务
            afterCompletion(transactionStatus, null);
            return result;
        } catch (Exception e) {
            // 回滚事务
            afterCompletion(transactionStatus, e);
            throw ExceptionsUtil.wrapRuntime(e);
        } finally {
            // 恢复原状态
            if (used == null) {
                TenantOrgContext.clearUseTenantOrg();
            } else {
                TenantOrgContext.setUseTenantOrg(used);
            }
            if (oldTenantOrgId == null) {
                TenantOrgContext.cleaTenantOrgId();
            } else {
                TenantOrgContext.setTenantOrgId(oldTenantOrgId);
            }
        }
    }

    private String generateTransactionName(Long tenantOrgId) {
        return TRANSACTION_NAME_PREFIX + (tenantOrgId == null ? "[default]" : "[" + tenantOrgId + "]");
    }

    private TransactionStatus buildTenantTransactionDefinition(String newTransactionName) {
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        definition.setName(newTransactionName);
        return transactionManager.getTransaction(definition);
    }

    private void afterCompletion(TransactionStatus transactionStatus, Exception exp) {
        if (transactionStatus == null || transactionStatus.isCompleted()) {
            return;
        }
        if (exp == null) {
            // 没有异常，则正常提交
            transactionManager.commit(transactionStatus);
            return;
        }

        //出现异常则回滚
        transactionManager.rollback(transactionStatus);
        log.warn("执行业务异常：", exp);
    }
}
