package com.elitescloud.boot.tenant.client.common;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.tenant.client.TenantClientProperties;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.elitescloud.boot.util.ValidateUtil;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;

/**
 * 租户相关Request工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/9
 */
@Log4j2
public class TenantRequestUtil {

    private static TenantClientProperties clientProperties;
    private static TenantProvider tenantProvider;
    private static RedisWrapper tenantRedisWrapper;

    protected TenantRequestUtil() {
    }

    /**
     * 解析租户域名
     *
     * @param request           请求
     * @return 当前请求的域名
     */
    public static String obtainTenantDomain(HttpServletRequest request) {
        String url = HttpServletUtil.obtainDomain(request);

        // 判断是否属于我们的子域名
        String defaultDomain = getClientProperties().getTenantDomain();
        if (StringUtils.hasText(defaultDomain)) {
            url = filterChildDomain(url, defaultDomain);
        }

        // 排除掉IP
        if (ValidateUtil.isIp(url)) {
            return null;
        }

        return url;
    }

    /**
     * 从请求信息中解析租户
     *
     * @param request request
     * @return 租户信息
     */
    public static SysTenantDTO obtainTenant(HttpServletRequest request) {
        // 优先从当前用户信息中获取
        var user = SecurityContextUtil.currentUser();
        if (user != null && user.getTenant() != null) {
            return user.getTenant();
        }

        try {
            return (SysTenantDTO) getTenantRedisWrapper().apply(() -> {
                // 从请求头获取
                String tenantId = request.getHeader(TenantConstant.HEADER_TENANT_ID);
                if (StringUtils.hasText(tenantId)) {
                    if (TenantConstant.DEFAULT_TENANT_ID.toString().equals(tenantId)) {
                        // 默认租户
                        return null;
                    }
                    var tenant = getTenantProvider().getById(Long.parseLong(tenantId));
                    return tenant.orElseThrow();
                }
                String tenantCode = request.getHeader(TenantConstant.HEADER_TENANT_CODE);
                if (StringUtils.hasText(tenantCode)) {
                    if (TenantConstant.DEFAULT_TENANT_ID.toString().equals(tenantId)) {
                        // 默认租户
                        return null;
                    }
                    var tenant = getTenantProvider().getByCode(tenantCode);
                    return tenant.orElseThrow();
                }

                // 根据域名转换
                if (StringUtils.hasText(getClientProperties().getTenantDomain())) {
                    String domain = obtainTenantDomain(request);
                    if (StringUtils.hasText(domain)) {
                        var tenant = getTenantProvider().getByDomain(domain);
                        if (tenant.isPresent()) {
                            log.debug("请求【{}】解析得租户：{}", request.getRequestURL().toString(), tenant.get().getTenantName());
                            return tenant.get();
                        }
                        log.debug("请求【{}】解析得域名【{}】未查询到租户信息", request.getRequestURL().toString(), domain);
                    }
                }
                return null;
            }, null);
        } catch (Exception e) {
            throw ExceptionsUtil.normalize(e, "获取当前租户失败");
        }
    }

    public static TenantClientProperties getClientProperties() {
        if (clientProperties == null) {
            clientProperties = SpringContextHolder.getObjectProvider(TenantClientProperties.class).getIfAvailable();
            Assert.state(clientProperties != null, "未启用租户");
        }
        return clientProperties;
    }

    public static TenantProvider getTenantProvider() {
        if (tenantProvider == null) {
            tenantProvider = SpringContextHolder.getObjectProvider(TenantProvider.class).getIfAvailable();
            Assert.state(tenantProvider != null, "未启用租户");
        }
        return tenantProvider;
    }

    public static RedisWrapper getTenantRedisWrapper() {
        if (tenantRedisWrapper == null) {
            tenantRedisWrapper = SpringContextHolder.getBean(RedisWrapper.class);
        }
        return tenantRedisWrapper;
    }

    private static String filterChildDomain(String domain, String defaultDomain) {
        if (domain == null) {
            return null;
        }
        if (domain.endsWith(defaultDomain)) {
            var end = domain.length() - defaultDomain.length() - 1;
            if (end <= 0) {
                return domain;
            }
            return domain.substring(0, end);
        }
        return domain;
    }
}
