package com.elitescloud.boot.tenant.client.support.impl;

import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.support.CloudtInterceptor;
import com.elitescloud.boot.tenant.client.TenantClientProperties;
import com.elitescloud.boot.tenant.client.common.TenantRequestUtil;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.util.CollectionUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 租户请求拦截器.
 * <p>
 * 主要用于设置租户上下文信息
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/26
 */
@Log4j2
public class TenantRequestInterceptor implements CloudtInterceptor {
    private final List<RequestMatcher> excludeRequestMatcher;

    public TenantRequestInterceptor(TenantClientProperties clientProperties) {
        excludeRequestMatcher = createExcludeRequestMatcher(clientProperties.getExcludePattern());
    }

    @Override
    public boolean preHandle(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler) throws Exception {
        // 判断是否是白名单过滤的
        if (isExclude(request)) {
            return true;
        }

        var tenant = TenantRequestUtil.obtainTenant(request);
        if (tenant != null) {
            TenantContextHolder.setCurrentTenant(tenant);
        }
        return true;
    }

    @Override
    public void afterCompletion(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler, Exception ex) throws Exception {
        TenantContextHolder.clearCurrentTenant();
    }

    private boolean isExclude(HttpServletRequest request) {
        if (excludeRequestMatcher.isEmpty()) {
            // 没有白名单
            return false;
        }

        return excludeRequestMatcher.stream().anyMatch(t -> t.matches(request));
    }

    private List<RequestMatcher> createExcludeRequestMatcher(Set<String> excludePath) {
        Set<String> paths = new HashSet<>(CloudtInterceptor.staticResourcePatter());
        if (!CollectionUtils.isEmpty(excludePath)) {
            paths.addAll(excludePath);
        }

        return paths.stream()
                .map(AntPathRequestMatcher::new)
                .collect(Collectors.toList());
    }
}
