package com.elitescloud.boot.tenant.client.common;

import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.wrapper.BaseCallbackWrapper;
import com.elitescloud.boot.wrapper.Isolatable;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;

import java.util.function.Supplier;

/**
 * 租户相关隔离代理公共类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/14
 */
@Log4j2
public abstract class AbstractTenantIsolateDelegate<T, R, E> implements BaseCallbackWrapper<T, R> {

    /**
     * 是否支持调用
     *
     * @param tenant 租户信息
     * @return 是否执行
     */
    protected abstract boolean supportApply(SysTenantDTO tenant);

    /**
     * 调用前的操作
     *
     * @param supplier 业务操作
     * @param tenant   租户
     * @return 如果不为null，则直接返回
     */
    protected abstract E beforeApply(Supplier<R> supplier, SysTenantDTO tenant);

    /**
     * 执行完后的回调
     *
     * @param result      执行结果
     * @param customParam beforeApply返回的结果
     * @param useDefault  是否使用了默认的
     * @param exp         异常
     */
    protected abstract void afterCompletion(R result, E customParam, boolean useDefault, Exception exp);

    @Override
    public R apply(Supplier<R> supplier, T param) {
        if (Isolatable.NONE.equals(param)) {
            // 不使用租户隔离
            return applyNot(supplier);
        }
        SysTenantDTO tenant = convertSubject(param);

        // 不支持的，直接调用后返回
        if (!supportApply(tenant)) {
            return supplier.get();
        }

        if (tenant == null || tenant.getType() == TenantType.OPERATION) {
            // 使用默认的
            return applyDefault(supplier, false);
        }

        // 使用指定租户的
        return applyTenant(supplier, tenant);
    }

    /**
     * 主体转租户对象
     *
     * @param subject 主体
     * @return 租户
     */
    private SysTenantDTO convertSubject(T subject) {
        if (subject == null) {
            return null;
        }
        if (subject instanceof SysTenantDTO) {
            return (SysTenantDTO) subject;
        } else if (subject instanceof GeneralUserDetails) {
            GeneralUserDetails user = (GeneralUserDetails) subject;
            return user.getTenant();
        } else if (subject instanceof SysUserDTO) {
            SysUserDTO user = (SysUserDTO) subject;
            return user.getSysTenantVO();
        } else if (subject instanceof Long) {
            return TenantClient.getTenant((Long) subject);
        } else if (subject instanceof String) {
            return TenantClient.getTenant(Long.parseLong((String) subject));
        } else {
            throw new IllegalArgumentException("未知参数主体类型");
        }
    }

    private R applyNot(Supplier<R> supplier) {
        boolean hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            // 已经是不隔离的
            return supplier.get();
        }

        TenantSession.setNoTenant();
        try {
            return applyDefault(supplier, true);
        } finally {
            TenantSession.clearNoTenant();
        }
    }

    /**
     * 使用默认的
     *
     * @param supplier 业务逻辑
     * @param forNot   是否不需要隔离的
     * @return 结果
     */
    private R applyDefault(Supplier<R> supplier, boolean forNot) {
        boolean hasDefault = TenantSession.getUseDefault();
        if (hasDefault && !forNot) {
            // 目前已是使用默认
            return supplier.get();
        }

        if (!forNot) {
            // 需要使用默认的
            TenantSession.setUseDefault();
        }
        E customParam = null;
        try {
            customParam = beforeApply(supplier, null);
            R result = supplier.get();

            executeCompletion(result, customParam, true, null);
            return result;
        } catch (Exception e) {
            executeCompletion(null, customParam, true, e);
            throw e;
        } finally {
            if (!forNot) {
                TenantSession.clearUseDefault();
            }
        }
    }

    private R applyTenant(Supplier<R> supplier, SysTenantDTO tenant) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }

        // 如果之前使用默认的，则先取消使用默认
        var hasUsedDefault = TenantSession.getUseDefault();
        if (hasUsedDefault) {
            TenantSession.clearUseDefault();
        }

        // 判断当前事务的租户与新的租户是否一致，如果一致，则直接执行
        if (!hasNot && !hasUsedDefault) {
            var currentTenant = TenantClient.getSessionTenant();
            if (currentTenant != null && currentTenant.getId().longValue() == tenant.getId()) {
                return supplier.get();
            }
        }

        // 如果之前设置了session的租户，则先更换
        var hasUsedTenant = TenantSession.getCurrentTenant();
        TenantSession.setCurrentTenant(tenant);

        E customParam = null;
        try {
            customParam = beforeApply(supplier, tenant);
            R result = supplier.get();

            executeCompletion(result, customParam, false, null);
            return result;
        } catch (Exception e) {
            log.error("业务异常：", e);
            executeCompletion(null, customParam, false, e);
            throw e;
        } finally {
            // 恢复原session 租户
            if (hasUsedTenant == null) {
                TenantSession.clearCurrentTenant();
            } else {
                TenantSession.setCurrentTenant(hasUsedTenant);
            }

            // 恢复原是否使用默认
            if (hasUsedDefault) {
                TenantSession.setUseDefault();
            }

            // 恢复不使用
            if (hasNot) {
                TenantSession.setNoTenant();
            }
        }
    }

    private void executeCompletion(R result, E customParam, boolean useDefault, Exception exp) {
        afterCompletion(result, customParam, useDefault, exp);
    }
}
