package com.elitescloud.boot.threadpool.support;

import com.elitescloud.boot.threadpool.common.ContextTransfer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 上下文信息传递代理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/30
 */
@Slf4j
public class ContextTransferDelegate {

    private static final List<ContextTransfer> CONTEXT_TRANSFER_FACTORY = new ArrayList<>();

    private final List<ContextTransfer> contextTransfers;
    private final Map<ContextTransfer, Object> contextMap = new HashMap<>();
    private static ApplicationContext applicationContext;

    private ContextTransferDelegate(List<ContextTransfer> contextTransfers) {
        this.contextTransfers = contextTransfers;
    }

    /**
     * 设置所有可用的上下文传递器
     *
     * @param contextTransfersNew 上下文传递器
     */
    private static void setContextTransferFactory(List<ContextTransfer> contextTransfersNew) {
        ContextTransferDelegate.CONTEXT_TRANSFER_FACTORY.clear();
        if (contextTransfersNew == null || contextTransfersNew.isEmpty()) {
            return;
        }

        ContextTransferDelegate.CONTEXT_TRANSFER_FACTORY.addAll(contextTransfersNew);
    }

    /**
     * 获取代理对象
     *
     * @return 代理对象
     */
    public static ContextTransferDelegate getInstance() {
        if (CONTEXT_TRANSFER_FACTORY.isEmpty()) {
            loadContextTransferFactory();
        }
        return new ContextTransferDelegate(CONTEXT_TRANSFER_FACTORY);
    }

    /**
     * 通过代理获取上下文信息
     */
    public void getContext() {
        if (CollectionUtils.isEmpty(contextTransfers)) {
            return;
        }

        for (ContextTransfer<?> contextTransfer : contextTransfers) {
            Object value = null;
            try {
                value = contextTransfer.getContext();
            } catch (Exception e) {
                log.error("上下文信息获取异常：", e);
                continue;
            }
            if (value != null) {
                contextMap.put(contextTransfer, value);
            }
        }
    }

    /**
     * 设置上下文信息
     */
    public void setContext() {
        if (contextMap.isEmpty()) {
            return;
        }

        for (Map.Entry<ContextTransfer, Object> entry : contextMap.entrySet()) {
            try {
                entry.getKey().setContext(entry.getValue());
            } catch (Exception e) {
                log.error("上下文信息设置异常：", e);
            }
        }
    }

    /**
     * 清除上下文
     */
    public void clearContext() {
        if (contextMap.isEmpty()) {
            return;
        }

        for (Map.Entry<ContextTransfer, Object> entry : contextMap.entrySet()) {
            try {
                entry.getKey().clearContext();
            } catch (Exception e) {
                log.error("上下文信息清除异常：", e);
            }
        }
    }

    public static void setApplicationContext(ApplicationContext applicationContext) {
        ContextTransferDelegate.applicationContext = applicationContext;
    }

    private static void loadContextTransferFactory() {
        Assert.notNull(applicationContext, "ApplicationContext还未初始化");
        var transferProvider = applicationContext.getBeanProvider(ContextTransfer.class);
        var transfers = transferProvider.stream().collect(Collectors.toList());
        setContextTransferFactory(transfers);
    }
}
