package com.elitescloud.boot.threadpool.common;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.dtp.common.config.ThreadPoolProperties;
import com.dtp.common.em.RejectedTypeEnum;
import com.dtp.core.DtpRegistry;
import com.dtp.core.support.ThreadPoolBuilder;
import com.dtp.core.support.wrapper.TaskWrapper;
import com.dtp.core.thread.DtpExecutor;
import com.dtp.core.thread.NamedThreadFactory;
import com.elitescloud.boot.threadpool.support.CloudtCoreTaskWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 线程池工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/21
 */
public class ThreadPoolHolder {
    private static final Logger LOG = LoggerFactory.getLogger(ThreadPoolHolder.class);

    private static final String THREAD_POOL_NAME = "cloudT-dtp-common";
    private static final String THREAD_PREFIX = "cloudT-common-";
    private static final int THREAD_MAX_SIZE = Runtime.getRuntime().availableProcessors();

    /**
     * 是否启用动态线程池管理
     */
    private static boolean dtpEnabled = true;

    protected ThreadPoolHolder() {
    }

    /**
     * 创建线程池
     *
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool() {
        return createThreadPool(null, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix) {
        return createThreadPool(threadPrefix, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize) {
        return createThreadPool(threadPrefix, coreSize, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @param maxSize      线程最多数量
     * @return 线程池
     */
    public static synchronized ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize, Integer maxSize) {
        return createThreadPool(null, threadPrefix, coreSize, maxSize);
    }

    /**
     * 创建线程池
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static synchronized ThreadPoolExecutor createThreadPool(String threadPoolName, String threadPrefix, Integer coreSize, Integer maxSize) {
        threadPoolName = CharSequenceUtil.blankToDefault(threadPoolName, THREAD_POOL_NAME);
        threadPrefix = CharSequenceUtil.blankToDefault(threadPrefix, THREAD_PREFIX);
        coreSize = ObjectUtil.defaultIfNull(coreSize, 1);
        maxSize = ObjectUtil.defaultIfNull(maxSize, THREAD_MAX_SIZE);

        DtpExecutor executor = null;
        if (existsRegistry(threadPoolName)) {
            executor = DtpRegistry.getDtpExecutor(threadPoolName);
        }

        ThreadPoolProperties propUpdate = null;
        if (executor == null) {
            LOG.info("create thread pool:{}, {}, {}, {}", threadPrefix, threadPoolName, coreSize, maxSize);
            executor = ThreadPoolBuilder.newBuilder()
                    .threadPoolName(threadPoolName)
                    .threadFactory(threadPrefix)
                    .corePoolSize(Objects.requireNonNullElse(coreSize, 1))
                    .maximumPoolSize(Objects.requireNonNullElse(maxSize, THREAD_MAX_SIZE))
                    .taskWrappers(taskWrappers())
                    .rejectedExecutionHandler(RejectedTypeEnum.ABORT_POLICY.getName())
                    .buildDynamic();
        } else {
            propUpdate = new ThreadPoolProperties();
            propUpdate.setCorePoolSize(Objects.requireNonNullElse(coreSize, 1));
            propUpdate.setMaximumPoolSize(Objects.requireNonNullElse(maxSize, THREAD_MAX_SIZE));
            propUpdate.setThreadNamePrefix(threadPrefix);
        }

        if (dtpEnabled) {
            if (propUpdate == null) {
                DtpRegistry.registerDtp(executor, "cloudt-common");
            } else {
                executor.setCorePoolSize(propUpdate.getCorePoolSize());
                executor.setMaximumPoolSize(propUpdate.getMaximumPoolSize());
                executor.setThreadFactory(new NamedThreadFactory(propUpdate.getThreadNamePrefix()));
                executor.setTaskWrappers(taskWrappers());
            }
        }
        return executor;
    }

    private static boolean existsRegistry(String threadPoolName) {
        if (!dtpEnabled) {
            return false;
        }
        return DtpRegistry.listAllDtpNames().contains(threadPoolName);
    }

    private static List<TaskWrapper> taskWrappers() {
        return List.of(new CloudtCoreTaskWrapper());
    }
}

