package com.elitescloud.boot.threadpool.common;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.threadpool.support.CloudtCoreTaskWrapper;
import org.dromara.dynamictp.common.em.RejectedTypeEnum;
import org.dromara.dynamictp.common.queue.VariableLinkedBlockingQueue;
import org.dromara.dynamictp.core.DtpRegistry;
import org.dromara.dynamictp.core.executor.DtpExecutor;
import org.dromara.dynamictp.core.executor.NamedThreadFactory;
import org.dromara.dynamictp.core.support.ExecutorWrapper;
import org.dromara.dynamictp.core.support.ThreadPoolBuilder;
import org.dromara.dynamictp.core.support.task.wrapper.TaskWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 线程池工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/21
 */
public class ThreadPoolHolder {
    private static final Logger LOG = LoggerFactory.getLogger(ThreadPoolHolder.class);

    public static final String THREAD_POOL_NAME = "cloudT-dtp-common";
    private static final String THREAD_PREFIX = "cloudT-common-";
    private static final int THREAD_MAX_SIZE = Runtime.getRuntime().availableProcessors();

    /**
     * 是否启用动态线程池管理
     */
    private static boolean dtpEnabled = true;

    protected ThreadPoolHolder() {
    }

    /**
     * 创建线程池
     *
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool() {
        return createThreadPool(null, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix) {
        return createThreadPool(threadPrefix, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize) {
        return createThreadPool(threadPrefix, coreSize, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @param maxSize      线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize, Integer maxSize) {
        return createThreadPool(null, threadPrefix, coreSize, maxSize);
    }

    /**
     * 创建线程池
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPoolName, String threadPrefix,
                                                      Integer coreSize, Integer maxSize) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, null);
    }

    /**
     * 创建线程池
     * <p>
     * common模式，适用于CPU密集型
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPoolName, String threadPrefix,
                                                      Integer coreSize, Integer maxSize, Integer queueCapacity) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity, false, false, false, false);
    }

    /**
     * 创建线程池
     * <p>
     * eager模式，适用于IO密集型
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPoolEager(String threadPoolName, String threadPrefix,
                                                           Integer coreSize, Integer maxSize, Integer queueCapacity) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity, true, false, false, false);
    }

    /**
     * 创建线程池
     * <p>
     * ordered模式，适用于IO密集型
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPoolOrdered(String threadPoolName, String threadPrefix,
                                                           Integer coreSize, Integer maxSize, Integer queueCapacity) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity, false, true, false, false);
    }

    /**
     * 创建线程池
     * <p>
     * priority模式，适用于根据任务优先级执行
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPoolPriority(String threadPoolName, String threadPrefix,
                                                             Integer coreSize, Integer maxSize, Integer queueCapacity) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity, false, false, false, true);
    }

    /**
     * 创建线程池
     * <p>
     * scheduled模式，适用定时任务
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPoolScheduled(String threadPoolName, String threadPrefix,
                                                              Integer coreSize, Integer maxSize, Integer queueCapacity) {
        return createThreadPool(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity, false, false, true, false);
    }

    /**
     * 创建线程池
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    private static synchronized ThreadPoolExecutor createThreadPool(String threadPoolName, String threadPrefix,
                                                                    Integer coreSize, Integer maxSize, Integer queueCapacity,
                                                                    boolean eager, boolean order, boolean scheduled, boolean priority) {
        threadPrefix = CharSequenceUtil.blankToDefault(threadPrefix, THREAD_PREFIX);
        threadPoolName = CharSequenceUtil.blankToDefault(threadPoolName, threadPrefix + THREAD_POOL_NAME);
        coreSize = ObjectUtil.defaultIfNull(coreSize, 1);
        maxSize = ObjectUtil.defaultIfNull(maxSize, THREAD_MAX_SIZE);

        // 判断是否已存在，存在则更新
        if (existsRegistry(threadPoolName)) {
            // 更新线程池
            updateExecutor(threadPoolName, threadPrefix, coreSize, maxSize, queueCapacity);
            return DtpRegistry.getDtpExecutor(threadPoolName);
        }

        // 不存在则创建
        var executorBuilder = ThreadPoolBuilder.newBuilder()
                .threadPoolName(threadPoolName)
                .threadFactory(threadPrefix)
                .corePoolSize(coreSize)
                .maximumPoolSize(maxSize)
                .queueCapacity(ObjectUtil.defaultIfNull(queueCapacity, 256))
                .taskWrappers(taskWrappers())
                .rejectedExecutionHandler(RejectedTypeEnum.CALLER_RUNS_POLICY.getName())
                .awaitTerminationSeconds(30);
        DtpExecutor executor = null;
        if (eager) {
            executor = executorBuilder.buildEager();
        } else if (order) {
            executor = executorBuilder.buildOrdered();
        } else if (scheduled) {
            executor = (DtpExecutor) executorBuilder.buildScheduled();
        } else if (priority) {
            executor = executorBuilder.buildPriority();
        } else {
            executor = executorBuilder.buildDynamic();
        }
        if (dtpEnabled) {
            DtpRegistry.registerExecutor(ExecutorWrapper.of(executor), "cloudt");
        }

        return executor;
    }

    private static void updateExecutor(String threadPoolName, String threadPrefix,
                                       int coreSize, int maxSize, Integer queueCapacity) {
        var executor = DtpRegistry.getExecutorWrapper(threadPoolName).getExecutor();
        executor.setCorePoolSize(coreSize);
        executor.setMaximumPoolSize(maxSize);

        if (executor instanceof DtpExecutor) {
            DtpExecutor dtpExecutor = (DtpExecutor) executor;
            dtpExecutor.setThreadFactory(new NamedThreadFactory(threadPrefix));
        }

        var queue = executor.getQueue();
        if (queue instanceof VariableLinkedBlockingQueue && queueCapacity != null) {
            int capacity = queue.size() + queue.remainingCapacity();
            if (!Objects.equals(capacity, queueCapacity)) {
                ((VariableLinkedBlockingQueue<Runnable>) queue).setCapacity(queueCapacity);
                executor.onRefreshQueueCapacity(queueCapacity);
            }
        }
    }


    private static boolean existsRegistry(String threadPoolName) {
        if (!dtpEnabled) {
            return false;
        }
        return DtpRegistry.getAllExecutorNames().contains(threadPoolName);
    }

    private static List<TaskWrapper> taskWrappers() {
        return List.of(new CloudtCoreTaskWrapper());
    }
}

