package com.elitescloud.boot.threadpool.support;

import com.elitescloud.boot.CloudtSpringContextHolder;
import com.elitescloud.boot.threadpool.config.ThreadPoolProperties;
import org.apache.skywalking.apm.toolkit.trace.RunnableWrapper;
import org.dromara.dynamictp.core.support.task.wrapper.TaskWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 任务包装器.
 *
 * @author Kaiser（wang shao）
 * @date 2023/4/27
 */
public class CloudtCoreTaskWrapper implements TaskWrapper {
    private static final Logger LOG = LoggerFactory.getLogger(CloudtCoreTaskWrapper.class);

    private ThreadPoolProperties threadPoolProperties;

    @Override
    public String name() {
        return "cloudt-core";
    }

    @Override
    public Runnable wrap(Runnable runnable) {
        // 调用者传递栈信息
        StackTraceElement[] stackTraceElements = null;
        try {
            if (threadPoolProperties == null) {
                threadPoolProperties = CloudtSpringContextHolder.getApplicationContext().getBeanProvider(ThreadPoolProperties.class).getIfAvailable();
            }
            boolean supportStackStrace = threadPoolProperties != null && Boolean.TRUE.equals(threadPoolProperties.getEnableStackTrace());
            stackTraceElements = supportStackStrace ? Thread.currentThread().getStackTrace() : new StackTraceElement[0];
        } catch (Exception e) {
            LOG.error("获取调用者堆栈信息异常：", e);
        }

        // 上下文信息
        var contextTransferDelegate = ContextTransferDelegate.getInstance();
        contextTransferDelegate.getContext();

        return new CloudtRunnable(RunnableWrapper.of(runnable), stackTraceElements, contextTransferDelegate);
    }

    static class CloudtRunnable implements Runnable {
        private final Runnable original;
        private final StackTraceElement[] stackTraceElements;
        private final ContextTransferDelegate contextTransferDelegate;

        public CloudtRunnable(Runnable original, StackTraceElement[] stackTraceElements, ContextTransferDelegate contextTransferDelegate) {
            this.original = original;
            this.stackTraceElements = stackTraceElements;
            this.contextTransferDelegate = contextTransferDelegate;
        }

        @Override
        public void run() {
            // 设置上下文信息
            contextTransferDelegate.setContext();

            try {
                original.run();
            } catch (Throwable throwable) {
                LOG.error("异步任务异常：", throwable);
                throw throwable;
            } finally {
                // 清除上下文信息
                contextTransferDelegate.clearContext();
            }
        }

        public Runnable getOriginal() {
            return original;
        }

        public StackTraceElement[] getStackTraceElements() {
            return stackTraceElements;
        }
    }
}
