package com.elitescloud.boot.threadpool.support;

import cn.hutool.core.util.ArrayUtil;
import org.dromara.dynamictp.core.aware.ExecutorAware;
import org.dromara.dynamictp.core.executor.DtpExecutor;
import org.dromara.dynamictp.core.support.task.runnable.DtpRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

/**
 * 动态线程池插件.
 *
 * @author Kaiser（wang shao）
 * @date 2024/9/3
 */
public class CloudtDtpAware implements ExecutorAware {
    private static final Logger logger = LoggerFactory.getLogger(CloudtDtpAware.class);

    /**
     * 当前运行中的线程
     */
    private static final Map<String, Map<String, RunnableInfo>> runningMap = new ConcurrentHashMap<>(64);

    @Override
    public int getOrder() {
        return 0;
    }

    @Override
    public String getName() {
        return "cloudt";
    }

    /**
     * 获取正在运行的线程池及其线程信息
     *
     * @return 线程池信息
     */
    public static Map<String, Map<String, RunnableInfo>> getRunners() {
        Map<String, Map<String, RunnableInfo>> result = new HashMap<>(runningMap.size());
        for (var entry : runningMap.entrySet()) {
            result.put(entry.getKey(), Collections.unmodifiableMap(entry.getValue()));
        }
        return result;
    }

    @Override
    public void beforeExecute(Executor executor, Thread thread, Runnable runnable) {
        ExecutorAware.super.beforeExecute(executor, thread, runnable);

        if (!(executor instanceof DtpExecutor) || !(runnable instanceof DtpRunnable)) {
            return;
        }
        DtpExecutor dtpExecutor = (DtpExecutor) executor;
        DtpRunnable dtpRunnable = (DtpRunnable) runnable;

        var runMap = runningMap.computeIfAbsent(dtpExecutor.getThreadPoolName(), k -> new ConcurrentHashMap<>());

        var key = thread.getName();
        if (key == null) {
            return;
        }
        runMap.put(key, new RunnableInfo(dtpRunnable, thread));
    }

    @Override
    public void afterExecute(Executor executor, Runnable runnable, Throwable e) {
        ExecutorAware.super.afterExecute(executor, runnable, e);

        if (!(executor instanceof DtpExecutor) || !(runnable instanceof DtpRunnable)) {
            return;
        }
        DtpExecutor dtpExecutor = (DtpExecutor) executor;

        var key = Thread.currentThread().getName();
        if (key == null) {
            return;
        }
        runningMap.computeIfAbsent(dtpExecutor.getThreadPoolName(), k -> new ConcurrentHashMap<>())
                .remove(key);
    }

    public static class RunnableInfo {
        private final DtpRunnable runnable;
        private final Thread thread;
        private final StackTraceElement[] stackTraceElements;
        private final LocalDateTime startTime;

        public RunnableInfo(DtpRunnable runnable, Thread thread) {
            this.runnable = runnable;
            this.thread = thread;
            this.stackTraceElements = obtainStackTraceElements(runnable);
            this.startTime = LocalDateTime.now();
        }

        public String getName() {
            return runnable.getOriginRunnable().getClass().getName();
        }

        public Long getId() {
            return thread.getId();
        }

        public List<String> getStackNames() {
            if (ArrayUtil.isEmpty(stackTraceElements)) {
                return Collections.emptyList();
            }

            return Arrays.stream(stackTraceElements)
                    .map(t -> t.getClassName() + "." + t.getMethodName() + "#" + t.getLineNumber())
                    .collect(Collectors.toList());
        }

        public LocalDateTime getStartTime() {
            return startTime;
        }

        private static StackTraceElement[] obtainStackTraceElements(DtpRunnable dtpRunnable) {
            Runnable runnable = dtpRunnable.getRunnable();
            if (runnable instanceof CloudtCoreTaskWrapper.CloudtRunnable) {
                return ((CloudtCoreTaskWrapper.CloudtRunnable) runnable).getStackTraceElements();
            }

            return new StackTraceElement[0];
        }
    }
}
