package com.elitescloud.boot.transaction;

import org.aspectj.lang.annotation.Aspect;
import org.springframework.aop.Advisor;
import org.springframework.aop.aspectj.AspectJExpressionPointcut;
import org.springframework.aop.support.DefaultPointcutAdvisor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.EnableTransactionManagement;
import org.springframework.transaction.interceptor.*;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 事务自动配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/11/06
 */
@Aspect
@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(prefix = TransactionProperties.CONFIG_PREFIX, name = "enable", havingValue = "true")
@EnableConfigurationProperties(TransactionProperties.class)
@EnableTransactionManagement
class CloudtTransactionAutoConfiguration {

    private final Set<String> defaultRequiredPrefix = Set.of("add", "insert", "create", "submit", "upsert", "update", "modify", "edit", "save", "delete", "remove", "batch", "set", "exec", "import");
    private final Set<String> defaultReadonlyPrefix = Set.of("get", "find", "query", "list", "page", "select", "count", "is", "exists", "all", "export");

    private final PlatformTransactionManager transactionManager;
    private final TransactionProperties transactionProperties;

    public CloudtTransactionAutoConfiguration(PlatformTransactionManager transactionManager, TransactionProperties transactionProperties) {
        this.transactionManager = transactionManager;
        this.transactionProperties = transactionProperties;
    }

    @Bean(value = "txAdvice")
    public TransactionInterceptor transactionInterceptor() {
        // 名称匹配
        NameMatchTransactionAttributeSource source = new NameMatchTransactionAttributeSource();
        source.setNameMap(obtainTransactionAttribute());

        // 添加事务拦截操作
        TransactionInterceptor transactionInterceptor = new TransactionInterceptor();
        transactionInterceptor.setTransactionManager(transactionManager);
        transactionInterceptor.setTransactionAttributeSource(source);

        return transactionInterceptor;
    }

    @Bean(value = "transactionAdviceAdvisor")
    public Advisor transactionAdviceAdvisor() {
        AspectJExpressionPointcut pointcut = new AspectJExpressionPointcut();
        pointcut.setExpression("execution(" + transactionProperties.getPointcut() + ")");

        return new DefaultPointcutAdvisor(pointcut, transactionInterceptor());
    }

    private Map<String, TransactionAttribute> obtainTransactionAttribute() {
        var readonly = new RuleBasedTransactionAttribute();
        readonly.setReadOnly(true);
        readonly.setPropagationBehavior(TransactionDefinition.PROPAGATION_NEVER);

        var required = new RuleBasedTransactionAttribute();
        required.setReadOnly(false);
        required.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRED);
        required.setTimeout((int) transactionProperties.getTimeout().toSeconds());
        required.getRollbackRules().add(new RollbackRuleAttribute(Exception.class));

        var setRequired = transactionProperties.getRequiredMethodPrefix();
        setRequired.addAll(defaultRequiredPrefix);
        setRequired.remove("*");

        var setReadonly = transactionProperties.getReadonlyMethodPrefix();
        setReadonly.addAll(defaultReadonlyPrefix);
        setReadonly.remove("*");

        Map<String, TransactionAttribute> attributeMap = new HashMap<>(64);
        for (var p : setRequired) {
            attributeMap.put(p.trim() + "*", required);
        }
        for (var p : setReadonly) {
            attributeMap.put(p.trim() + "*", readonly);
        }
        // 其它默认都是只读的
        attributeMap.put("*", readonly);

        return attributeMap;
    }
}
