package com.elitescloud.boot.web.exception;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.exception.CustomExceptionTranslate;
import com.elitescloud.boot.web.config.CloudtExceptionProperties;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.ResourceAccessException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 全局异常处理
 *
 * @author Mir
 * @date 2020/6/22
 */
@Slf4j
@ControllerAdvice
@ConditionalOnBean(CloudtExceptionProperties.class)
public class GlobalExceptionHandler {

    private final CloudtExceptionProperties exceptionProperties;
    private List<CustomExceptionTranslate> exceptionTranslates;

    public GlobalExceptionHandler(CloudtExceptionProperties exceptionProperties) {
        this.exceptionProperties = exceptionProperties;
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    @ResponseBody
    public ApiResult<String> handleHttpMessageNotReadableException(HttpServletRequest request, HttpServletResponse response,
                                                                   HttpMessageNotReadableException e) {
        String errorNo = printError(request, response, e, "请求数据格式有误");
        return ApiResult.fail(ApiCode.PARAMETER_PARSE_EXCEPTION, errorNo, stackTrace(e), null, null);
    }

    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public ApiResult<String> handleHttpRequestMethodNotSupportedException(HttpServletRequest request, HttpServletResponse response,
                                                                          HttpRequestMethodNotSupportedException e) {
        String errorNo = printError(request, response, e, "请求方式有误");
        return ApiResult.fail(ApiCode.METHOD_NOT_SUPPORT, errorNo, stackTrace(e), null, null);
    }

    @ExceptionHandler(HttpMediaTypeNotAcceptableException.class)
    @ResponseBody
    public ApiResult<String> handleHttpMediaTypeNotAcceptableException(HttpServletRequest request, HttpServletResponse response,
                                                                       HttpMediaTypeNotAcceptableException e) {
        String errorNo = printError(request, response, e, "请求媒体类型有误");
        String errorMsg = CollUtil.isEmpty(e.getSupportedMediaTypes()) ? "不支持的Content-Type" :
                "仅支持以下Content-Type:" + MediaType.toString(e.getSupportedMediaTypes())
        ;

        return ApiResult.fail(ApiCode.UNSUPPORTED_MEDIA_TYPE, errorNo, stackTrace(e), null, errorMsg);
    }

    @ExceptionHandler(HttpClientErrorException.class)
    @ResponseBody
    public ApiResult<String> handleHttpClientErrorException(HttpServletRequest request, HttpServletResponse response,
                                                            HttpClientErrorException e) {
        String errorNo = printError(request, response, e, "客户端请求异常");

        String msg = null;
        if (e.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            msg = "暂未登录或token已经过期";
            response.setStatus(HttpStatus.UNAUTHORIZED.value());
        }

        return ApiResult.fail(e.getRawStatusCode(), errorNo, stackTrace(e), null, msg);
    }

    /**
     * 参数校验异常
     *
     * @param exception 异常
     * @return 结果
     */
    @ExceptionHandler({IllegalArgumentException.class, MethodArgumentNotValidException.class, MissingServletRequestParameterException.class,
            ConstraintViolationException.class, ValidationException.class})
    @ResponseBody
    public ApiResult<String> handleArgumentInvalidException(HttpServletRequest request, HttpServletResponse response,
                                                            Exception exception) {
        String errorNo = printError(request, response, exception, "参数校验不通过");

        if (exception instanceof IllegalArgumentException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, exception.getMessage());
        }

        if (exception instanceof MethodArgumentNotValidException) {
            String msg = ((MethodArgumentNotValidException) exception).getBindingResult().getAllErrors().stream()
                    .map(ObjectError::getDefaultMessage)
                    .collect(Collectors.joining(";"));
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, msg);
        }

        if (exception instanceof MissingServletRequestParameterException) {
            String field = ((MissingServletRequestParameterException) exception).getParameterName();
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, field + "为空");
        }

        if (exception instanceof ConstraintViolationException) {
            String msg = ((ConstraintViolationException) exception).getConstraintViolations().stream()
                    .map(ConstraintViolation::getMessage)
                    .collect(Collectors.joining(";"));
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, msg);
        }

        if (exception instanceof ValidationException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, exception.getMessage());
        }

        return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, null, ExceptionUtil.getRootCause(exception).getMessage());
    }

    @ExceptionHandler(BindException.class)
    @ResponseBody
    public ApiResult<String> handleBindException(HttpServletRequest request, HttpServletResponse response, BindException e) {
        String errorNo = printError(request, response, e, "请求参数有误");

        return ApiResult.fail(ApiCode.PARAMETER_PARSE_EXCEPTION, errorNo, stackTrace(e), null, "请求参数有误");
    }
    // 客户端请求错误 end

    // 服务端异常
    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public ApiResult<String> handleBusinessException(HttpServletRequest request, HttpServletResponse response, BusinessException e) {
        var result = customTranslate(request, response, e);
        if (result != null) {
            return result;
        }

        String errorNo = printError(request, response, e, "业务异常");
        if (e.getCode() != null) {
            return ApiResult.fail(e.getCode(), errorNo, stackTrace(e), null, e.getMessage());
        }

        ApiCode apiCode = e.getApiCode();
        if (apiCode == null) {
            apiCode = ApiCode.FAIL;
        } else if (apiCode.getCode() == HttpStatus.UNAUTHORIZED.value()) {
            response.setStatus(HttpStatus.UNAUTHORIZED.value());
        }
        return ApiResult.fail(apiCode, errorNo, stackTrace(e), null, e.getMessage());
    }

    @ExceptionHandler(ResourceAccessException.class)
    @ResponseBody
    public ApiResult<String> handleResourceAccessException(HttpServletRequest request, HttpServletResponse response, ResourceAccessException e) {
        String errorNo = printError(request, response, e, "请求资源异常");
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, errorNo, stackTrace(e), null, null);
    }

    @ExceptionHandler({Exception.class})
    @ResponseBody
    public ApiResult<String> handleException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        var rootCause = ExceptionUtil.getRootCause(e);

        if (rootCause instanceof BusinessException) {
            return handleBusinessException(request, response, (BusinessException) rootCause);
        }
        if (rootCause instanceof IllegalArgumentException) {
            return handleArgumentInvalidException(request, response, (Exception) rootCause);
        }

        var result = customTranslate(request, response, e);
        if (result != null) {
            return result;
        }

        String errorNo = printError(request, response, e, "系统异常");
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, errorNo, stackTrace(e), null, null);
    }

    private ApiResult<String> customTranslate(HttpServletRequest request, HttpServletResponse response, Throwable e) {
        if (exceptionTranslates == null) {
            // 初始化翻译器
            var translates = SpringContextHolder.getObjectProvider(CustomExceptionTranslate.class);
            exceptionTranslates = new ArrayList<>(8);
            for (CustomExceptionTranslate translate : translates) {
                exceptionTranslates.add(translate);
            }
        }
        if (exceptionTranslates.isEmpty()) {
            return null;
        }

        ApiResult<String> apiResult;
        for (CustomExceptionTranslate exceptionTranslate : exceptionTranslates) {
            if (exceptionTranslate.support(e)) {
                apiResult = exceptionTranslate.translate(e);
                if (apiResult != null) {
                    if (apiResult.getCode() == ApiCode.UNAUTHORIZED.getCode()) {
                        response.setStatus(apiResult.getCode());
                    }
                    // 自定义异常翻译已处理，设置错误号
                    if (apiResult.getErrorNo() == null) {
                        String errorNo = printError(request, response, e, exceptionTranslate.getClass().getName() + "处理异常");
                        apiResult.setErrorNo(errorNo);
                    }
                    if (apiResult.getData() == null) {
                        apiResult.setData(stackTrace(e));
                    }
                    return apiResult;
                }
            }
        }
        return null;
    }

    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");

    private String printError(HttpServletRequest request, HttpServletResponse response, Throwable throwable, String description) {
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        String errorNo = FORMATTER.format(LocalDateTime.now());
        log.warn("请求地址：{}, {}", request.getRequestURL(), request.getQueryString());
        String msg = description + "，错误号：" + errorNo;
        log.warn(msg, throwable);

        return errorNo;
    }

    private String stackTrace(Throwable throwable) {
        if (!exceptionToData()) {
            // 不写入异常信息
            return null;
        }
        return ExceptionUtil.stacktraceToString(throwable, -1);
    }

    private boolean exceptionToData() {
        var toData = exceptionProperties.getGlobal().getExceptionDetailToData();
        return toData != null && toData;
    }
}
