package com.elitescloud.boot.web.config.filter;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ClassUtil;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.support.CloudtInterceptor;
import com.elitescloud.boot.support.CloudtStarterTool;
import com.elitescloud.boot.util.CloudtBootUtil;
import com.elitescloud.boot.web.common.RepeatKeyGenerator;
import com.elitescloud.boot.web.config.WebProperties;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.web.method.HandlerMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.TimeUnit;

/**
 * 重复请求拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/31
 */
@Slf4j
public class WebRequestRepeatInterceptor implements CloudtInterceptor {

    private final RepeatKeyGenerator repeatKeyGenerator;
    private final RedisUtils redisUtils;
    private final WebProperties webProperties;

    public WebRequestRepeatInterceptor(RedisUtils redisUtils,
                                       WebProperties webProperties) {
        this.repeatKeyGenerator = this.loadRepeatKeyGenerator(webProperties.getRepeatRequest().getKeyGenerator());
        this.redisUtils = redisUtils;
        this.webProperties = webProperties;

        log.info("WebRequestRepeatInterceptor: {}, {}", webProperties.getRepeatRequest().getStrategy(), webProperties.getRepeatRequest().getIntervalMillis());
    }

    @Override
    public int order() {
        return Integer.MIN_VALUE + 1;
    }

    @Override
    public boolean preHandle(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler) throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            log.debug("重复请求拦截跳过：{}", request.getRequestURI());
            return true;
        }

        // 判断是否已禁用
        var config = webProperties.getRepeatRequest();
        if (Boolean.FALSE.equals(config.isEnabled()) || config.getStrategy() == WebProperties.RepeatInterceptStrategy.NEVER) {
            return true;
        }

        // 获取方法上的注解
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        var businessObjectOperation = handlerMethod.getMethodAnnotation(BusinessObjectOperation.class);
        if (config.getStrategy() == WebProperties.RepeatInterceptStrategy.AUTO) {
            // 未设置注解 或 允许重复请求，则不拦截
            if (businessObjectOperation == null || businessObjectOperation.allowRepeatRequest()) {
                return true;
            }
        }

        // 时间间隔
        int inverval = config.getIntervalMillis();
        if (businessObjectOperation != null && businessObjectOperation.repeatIntervalMillis() > 0) {
            inverval = businessObjectOperation.repeatIntervalMillis();
        }
        if (inverval < 0) {
            inverval = 1000;
        }

        // 判断是否允许请求
        if (allow(request, inverval)) {
            return true;
        }

        // 不允许请求
        response.setStatus(config.getResponseHttpStatus());
        HttpServletUtil.writeJson(response, ApiResult.fail(ApiCode.REQUEST_TOO_MANY, config.getResponseMsg()));

        return false;
    }

    private boolean allow(HttpServletRequest request, int interval) {
        var key = repeatKeyGenerator.generate(request);
        if (key == null) {
            // 未生成有效key，判断是否放行
            if (Boolean.TRUE.equals(webProperties.getRepeatRequest().isAllowKeyEmpty())) {
                log.warn("未生成有效请求标识，允许请求重复：{}", request.getRequestURI());
                return true;
            }
            return false;
        }

        var result = redisUtils.getRedisTemplate().opsForValue().setIfAbsent(key, "true", interval, TimeUnit.MILLISECONDS);
        log.info("repeat request:{}, {}", key, result);
        if (result == null) {
            log.warn("当前Redis不支持setIfAbsent操作！");
            return true;
        }
        return result;
    }

    private RepeatKeyGenerator loadRepeatKeyGenerator(String className) {
        if (CharSequenceUtil.isBlank(className)) {
            throw new IllegalArgumentException("未配置有效的web重复调用判断key生成器：" + WebProperties.CONFIG_PREFIX + ".repeat-request.key-generator");
        }

        Class<?> keyGeneratorClass = null;
        try {
            keyGeneratorClass = Class.forName(className);
        } catch (ClassNotFoundException e) {
            log.error("Key生成器不存在：{}", className, e);
            throw new IllegalStateException(e);
        }

        if (!RepeatKeyGenerator.class.isAssignableFrom(keyGeneratorClass)) {
            throw new IllegalStateException(className + "未实现接口：" + RepeatKeyGenerator.class.getName());
        }

        Constructor<RepeatKeyGenerator> constructorNoArgs = null;
        try {
            constructorNoArgs = (Constructor<RepeatKeyGenerator>) keyGeneratorClass.getConstructor();
        } catch (Exception e) {
            log.error("{}加载无参构造方法：", className, e);
            throw new RuntimeException(e);
        }
        try {
            return constructorNoArgs.newInstance();
        } catch (Exception e) {
            log.error("初始化{}异常：", className, e);
            throw new RuntimeException(e);
        }
    }
}
