package com.elitescloud.boot.web.config.filter;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.BusinessParamValueKeyGenerator;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.support.CloudtInterceptor;
import com.elitescloud.boot.util.CloudtSpelExpressionEvaluator;
import com.elitescloud.boot.web.common.RepeatKeyGenerator;
import com.elitescloud.boot.web.config.WebProperties;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.annotation.RequestBodyAdvice;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 重复请求拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/31
 */
@Slf4j
public class WebRequestRepeatInterceptor implements CloudtInterceptor, RequestBodyAdvice {
    private final RepeatKeyGenerator repeatKeyGenerator;
    private final RedisUtils redisUtils;
    private final WebProperties webProperties;

    private final ThreadLocal<RequestObject> requestObjectHolder = new ThreadLocal<>();
    private final Map<Class<? extends BusinessParamValueKeyGenerator>, BusinessParamValueKeyGenerator> businessKeyGeneratorInstanceMap = new HashMap<>();

    public WebRequestRepeatInterceptor(RedisUtils redisUtils,
                                       WebProperties webProperties) {
        this.repeatKeyGenerator = this.loadRepeatKeyGenerator(webProperties.getRepeatRequest().getKeyGenerator());
        this.redisUtils = redisUtils;
        this.webProperties = webProperties;

        log.info("WebRequestRepeatInterceptor: {}, {}", webProperties.getRepeatRequest().getStrategy(), webProperties.getRepeatRequest().getIntervalMillis());
    }

    @Override
    public int order() {
        return Integer.MIN_VALUE + 1;
    }

    @Override
    public boolean preHandle(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler) throws Exception {
        if (!(handler instanceof HandlerMethod)) {
            log.debug("重复请求拦截跳过：{}", request.getRequestURI());
            return true;
        }

        // 判断是否已禁用
        var config = webProperties.getRepeatRequest();
        if (Boolean.FALSE.equals(config.isEnabled()) || config.getStrategy() == WebProperties.RepeatInterceptStrategy.NEVER) {
            return true;
        }

        // 获取方法上的注解
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        var businessObjectOperation = handlerMethod.getMethodAnnotation(BusinessObjectOperation.class);
        if (config.getStrategy() == WebProperties.RepeatInterceptStrategy.AUTO) {
            // 未设置注解 或 允许重复请求，则不拦截
            if (businessObjectOperation == null || businessObjectOperation.allowRepeatRequest()) {
                return true;
            }
        }

        // 判断是否需要根据业务参数key
        if (this.supportBusinessParams(request, response, handlerMethod, businessObjectOperation)) {
            return true;
        }

        // 根据请求生成key
        var key = repeatKeyGenerator.generate(request, handlerMethod);
        // 判断是否允许请求
        if (this.allow(request, key, businessObjectOperation)) {
            return true;
        }

        // 不允许请求
        response.setStatus(config.getResponseHttpStatus());
        HttpServletUtil.writeJson(response, ApiResult.fail(ApiCode.REQUEST_TOO_MANY, config.getResponseMsg()));

        return false;
    }

    @Override
    public boolean supports(@NonNull MethodParameter methodParameter, @NonNull Type targetType, @NonNull Class<? extends HttpMessageConverter<?>> converterType) {
        return requestObjectHolder.get() != null;
    }

    @Override
    public HttpInputMessage beforeBodyRead(@NonNull HttpInputMessage inputMessage, @NonNull MethodParameter parameter, @NonNull Type targetType, @NonNull Class<? extends HttpMessageConverter<?>> converterType) throws IOException {
        return inputMessage;
    }

    @Override
    public Object afterBodyRead(@NonNull Object body, @NonNull HttpInputMessage inputMessage, @NonNull MethodParameter parameter, @NonNull Type targetType, @NonNull Class<? extends HttpMessageConverter<?>> converterType) {
        var requestObject = requestObjectHolder.get();
        if (requestObject == null) {
            return body;
        }

        var key = generateRequestKeyByParams(requestObject, body);
        if (!this.allow(requestObject.getRequest(), key, requestObject.getBusinessObjectOperation())) {
            throw new BusinessException(ApiCode.REQUEST_TOO_MANY, webProperties.getRepeatRequest().getResponseMsg());
        }
        return body;
    }

    @Override
    public Object handleEmptyBody(Object body, @NonNull HttpInputMessage inputMessage, @NonNull MethodParameter parameter, @NonNull Type targetType, @NonNull Class<? extends HttpMessageConverter<?>> converterType) {
        return body;
    }

    @Override
    public void afterCompletion(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler, Exception ex) throws Exception {
        requestObjectHolder.remove();
    }

    private boolean supportBusinessParams(HttpServletRequest request, HttpServletResponse response, HandlerMethod handlerMethod, BusinessObjectOperation businessObjectOperation) {
        if (businessObjectOperation == null) {
            return false;
        }

        RequestObject requestObject = new RequestObject(request, response, handlerMethod);
        requestObject.setBusinessObjectOperation(businessObjectOperation);

        // 优先根据业务参数key
        if (StringUtils.hasText(businessObjectOperation.businessParamValueKey())) {
            requestObject.setBusinessValueKey(businessObjectOperation.businessParamValueKey());
            requestObjectHolder.set(requestObject);
            return true;
        }
        // 其次根据自定义参数生成器
        var businessKeyGenerator = businessObjectOperation.businessParamValueKeyGenerator();
        if (businessKeyGenerator != null && businessKeyGenerator != BusinessParamValueKeyGenerator.class) {
            Assert.isTrue(!businessKeyGenerator.isInterface() && BusinessParamValueKeyGenerator.class.isAssignableFrom(businessKeyGenerator),
                    "businessParamValueKeyGenerator必须是实现BusinessParamValueKeyGenerator的类");
            requestObject.setBusinessValueKeyGenerator(businessKeyGenerator);
            requestObjectHolder.set(requestObject);
            return true;
        }

        return false;
    }

    private String generateRequestKeyByParams(RequestObject requestObject, Object requestBody) {
        // 获取参数
        Object[] args = HttpServletUtil.getControllerParameters(requestObject.getHandlerMethod(), requestObject.getRequest(),
                requestObject.getResponse(), requestBody);

        String key = null;
        if (StringUtils.hasText(requestObject.getBusinessValueKey())) {
            // spel表达式解析
            Object keyObj = null;
            try {
                var evaluator = CloudtSpelExpressionEvaluator.getInstance();
                var evaluateContext = evaluator.createEvaluationContext(requestObject.getHandlerMethod().getBean(), requestObject.getHandlerMethod().getMethod(), args);
                keyObj = evaluator.parseExpression(requestObject.getBusinessValueKey(), evaluateContext);
            } catch (Exception e) {
                throw new IllegalStateException("解析业务参数key异常：" + requestObject.getBusinessValueKey(), e);
            }
            key = keyObj == null ? null : keyObj.toString();
        } else if (requestObject.getBusinessValueKeyGenerator() != null) {
            // 自定义生成器
            var generator = businessKeyGeneratorInstanceMap.get(requestObject.getBusinessValueKeyGenerator());
            if (generator == null) {
                // 不存在则初始化
                try {
                    var constructor = requestObject.getBusinessValueKeyGenerator().getConstructor();
                    generator = constructor.newInstance();
                } catch (Exception e) {
                    throw new IllegalStateException("初始化业务参数key的生成器异常：" + requestObject.getBusinessValueKeyGenerator().getName(), e);
                }
                businessKeyGeneratorInstanceMap.put(requestObject.getBusinessValueKeyGenerator(), generator);
            }
            key = generator.generate(requestObject.getRequest(), requestObject.getHandlerMethod(), args);
        } else {
            log.error("根据参数生成请求唯一标识异常");
        }

        return key;
    }

    private boolean allow(HttpServletRequest request, String key, BusinessObjectOperation businessObjectOperation) {
        if (key == null) {
            // 未生成有效key，判断是否放行
            if (Boolean.TRUE.equals(webProperties.getRepeatRequest().isAllowKeyEmpty())) {
                log.warn("未生成有效请求标识，允许请求重复：{}", request.getRequestURI());
                return true;
            }
            return false;
        }

        // 时间间隔
        int interval = this.obtainInterval(businessObjectOperation);

        // 更新缓存中的key
        var result = redisUtils.getRedisTemplate().opsForValue().setIfAbsent(key, "true", interval, TimeUnit.MILLISECONDS);
        log.info("repeat request:{}, {}", key, result);
        if (result == null) {
            log.warn("当前Redis不支持setIfAbsent操作！");
            return true;
        }
        return result;
    }

    private int obtainInterval(BusinessObjectOperation businessObjectOperation) {
        int inverval = webProperties.getRepeatRequest().getIntervalMillis();
        if (businessObjectOperation != null && businessObjectOperation.repeatIntervalMillis() > 0) {
            inverval = businessObjectOperation.repeatIntervalMillis();
        }
        if (inverval < 0) {
            inverval = 1000;
        }

        return inverval;
    }

    private RepeatKeyGenerator loadRepeatKeyGenerator(String className) {
        if (CharSequenceUtil.isBlank(className)) {
            throw new IllegalArgumentException("未配置有效的web重复调用判断key生成器：" + WebProperties.CONFIG_PREFIX + ".repeat-request.key-generator");
        }

        Class<?> keyGeneratorClass = null;
        try {
            keyGeneratorClass = Class.forName(className);
        } catch (ClassNotFoundException e) {
            log.error("Key生成器不存在：{}", className, e);
            throw new IllegalStateException(e);
        }

        if (!RepeatKeyGenerator.class.isAssignableFrom(keyGeneratorClass)) {
            throw new IllegalStateException(className + "未实现接口：" + RepeatKeyGenerator.class.getName());
        }

        Constructor<RepeatKeyGenerator> constructorNoArgs = null;
        try {
            constructorNoArgs = (Constructor<RepeatKeyGenerator>) keyGeneratorClass.getConstructor();
        } catch (Exception e) {
            log.error("{}加载无参构造方法：", className, e);
            throw new RuntimeException(e);
        }
        try {
            return constructorNoArgs.newInstance();
        } catch (Exception e) {
            log.error("初始化{}异常：", className, e);
            throw new RuntimeException(e);
        }
    }

    @Getter
    @Setter
    static class RequestObject {
        private final HttpServletRequest request;
        private final HttpServletResponse response;
        private final HandlerMethod handlerMethod;

        private BusinessObjectOperation businessObjectOperation;
        private String businessValueKey;
        private Class<? extends BusinessParamValueKeyGenerator> businessValueKeyGenerator;

        public RequestObject(HttpServletRequest request, HttpServletResponse response, HandlerMethod handlerMethod) {
            this.request = request;
            this.response = response;
            this.handlerMethod = handlerMethod;
        }
    }
}
