package com.elitescloud.boot.webservice.support.cxf;

import com.elitescloud.cloudt.common.base.annotation.webservice.WebServiceConsumer;
import com.elitescloud.boot.webservice.support.AbstractServiceConsumerConfig;
import lombok.extern.log4j.Log4j2;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.jaxws.JaxWsProxyFactoryBean;
import org.apache.cxf.jaxws.endpoint.dynamic.JaxWsDynamicClientFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.jws.WebMethod;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/11
 */
@ConditionalOnClass(JaxWsProxyFactoryBean.class)
@Log4j2
public class CxfServiceConsumerConfig extends AbstractServiceConsumerConfig {

    @Override
    public Object buildConsumer(@NonNull Class<?> serviceClass, String beanName, @NonNull WebServiceConsumer consumer) {
        if (consumer.javaServer()) {
            // java应用，支持静态代理
            return buildStaticProxy(serviceClass, beanName, consumer);
        }

        return buildDynamicProxy(serviceClass, beanName, consumer);
    }

    private Object buildDynamicProxy(Class<?> serviceClass, String beanName, WebServiceConsumer consumer) {
        // 创建客户端
        JaxWsDynamicClientFactory clientFactory = JaxWsDynamicClientFactory.newInstance();
        Client client = clientFactory.createClient(getAddress(consumer, beanName));

        Map<String, String> methodNameCache = new HashMap<>(128);
        InvocationHandler handler = (proxy, method, args) -> {
            // 解析调用的方法名
            var tempName = serviceClass.getName() + "#" + method.getName();
            String methodName = methodNameCache.computeIfAbsent(tempName, k -> {
                // 优先从注解上获取
                var webMethod = method.getAnnotation(WebMethod.class);
                if (webMethod != null && StringUtils.hasText(webMethod.operationName())) {
                    return webMethod.operationName();
                }

                // 如果没有注解，则使用方法名
                return method.getName();
            });

            return client.invoke(methodName, args);
        };

        return Proxy.newProxyInstance(serviceClass.getClassLoader(), new Class[]{serviceClass}, handler);
    }

    private Object buildStaticProxy(Class<?> serviceClass, String beanName, WebServiceConsumer consumer) {
        JaxWsProxyFactoryBean factoryBean = new JaxWsProxyFactoryBean();
        factoryBean.setServiceClass(serviceClass);
        String address = getAddress(consumer, beanName);
        factoryBean.setAddress(getAddress(consumer, beanName));

        log.info("WebService静态代理客户端【{}】绑定地址：{}", beanName, address);

        return factoryBean.create();
    }

    private String getAddress(WebServiceConsumer consumer, String beanName) {
        if (StringUtils.hasText(consumer.addressKey())) {
            String address = getConfigAddress(consumer.addressKey());
            Assert.hasText(address, consumer.addressKey() + "未配置");

            return address;
        }

        Assert.hasText(consumer.address(), beanName + "的addressKey()和address()至少配置一个");
        return consumer.address();
    }
}
