package com.elitescloud.boot.websocket.config;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.websocket.CloudtWebSocketHandler;
import com.elitescloud.boot.websocket.CloudtWebSocketTemplate;
import com.elitescloud.boot.websocket.endpoint.EndpointPathParamConvert;
import com.elitescloud.boot.websocket.handler.WebSocketMsgPublishable;
import com.elitescloud.boot.websocket.support.AuthHandshakeInterceptor;
import com.elitescloud.boot.websocket.support.CloudtWebSocketHandlerDelegate;
import com.elitescloud.boot.websocket.endpoint.WebSocketEndpoint;
import com.elitescloud.boot.websocket.support.redis.RedisWebSocketMsgPublishable;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.actuate.endpoint.annotation.EndpointConverter;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.lang.NonNull;
import org.springframework.web.socket.config.annotation.EnableWebSocket;
import org.springframework.web.socket.config.annotation.WebSocketConfigurer;
import org.springframework.web.socket.config.annotation.WebSocketHandlerRegistry;
import org.springframework.web.socket.server.standard.ServletServerContainerFactoryBean;

/**
 * websocket自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/30
 */
@EnableConfigurationProperties(CloudtWebSocketProperties.class)
@AutoConfigureAfter(CloudtWebSocketRedisAutoConfiguration.class)
@EnableWebSocket
@Slf4j
public class CloudtWebSocketAutoConfiguration implements WebSocketConfigurer {

    private final ObjectProvider<CloudtWebSocketHandler> handlerObjectProvider;
    private final UserDetailProvider userDetailProvider;

    public CloudtWebSocketAutoConfiguration(ObjectProvider<CloudtWebSocketHandler> handlerObjectProvider,
                                            UserDetailProvider userDetailProvider) {
        this.handlerObjectProvider = handlerObjectProvider;
        this.userDetailProvider = userDetailProvider;
    }

    @Override
    public void registerWebSocketHandlers(@NonNull WebSocketHandlerRegistry registry) {
        for (CloudtWebSocketHandler handler : handlerObjectProvider) {
            Assert.notEmpty(handler.paths(), handler.getClass().getName() + "的paths为空");
            registry.addHandler(new CloudtWebSocketHandlerDelegate(handler, userDetailProvider), handler.paths())
                    .addInterceptors(new AuthHandshakeInterceptor(userDetailProvider))
                    .setAllowedOrigins(handler.allowedOrigins())
                    .setAllowedOriginPatterns(handler.allowedOriginPatterns())
            ;
            log.info("registered WebSocketHandler：{}", handler.getClass().getName());
        }
    }

    @Bean
    public ServletServerContainerFactoryBean servletServerContainerFactoryBean(CloudtWebSocketProperties properties) {
        ServletServerContainerFactoryBean container = new ServletServerContainerFactoryBean();

        if (properties.getMaxTextMessageBufferSize() != null) {
            container.setMaxTextMessageBufferSize((int) properties.getMaxTextMessageBufferSize().toBytes());
        }
        if (properties.getMaxBinaryMessageBufferSize() != null) {
            container.setMaxBinaryMessageBufferSize((int) properties.getMaxBinaryMessageBufferSize().toBytes());
        }
        if (properties.getAsyncSendTimeout() != null) {
            container.setAsyncSendTimeout(properties.getAsyncSendTimeout().toMillis());
        }
        if (properties.getMaxSessionIdleTimeout() != null) {
            container.setMaxSessionIdleTimeout(properties.getMaxSessionIdleTimeout().toMillis());
        }
        return container;
    }

    @Bean
    public CloudtWebSocketTemplate cloudtWebSocketTemplate(ObjectProvider<WebSocketMsgPublishable> msgPublishables) {
        return new CloudtWebSocketTemplate(msgPublishables.getIfAvailable());
    }

    @Bean
    public WebSocketMsgPublishable redisWebSocketMsgPublishable(RedisUtils redisUtils) {
        return new RedisWebSocketMsgPublishable(redisUtils);
    }

    @Bean
    public WebSocketEndpoint webSocketEndpoint() {
        return new WebSocketEndpoint();
    }

    @Bean
    @EndpointConverter
    public EndpointPathParamConvert endpointPathParamConvert() {
        return new EndpointPathParamConvert();
    }
}
