package com.elitescloud.boot.xxljob;

import cn.hutool.core.util.StrUtil;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.core.tenant.support.TenantDataIsolateProvider;
import com.xxl.job.core.executor.impl.XxlJobSpringExecutor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

/**
 * xxl-job配置.
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/6
 */
@Log4j2
@ConditionalOnProperty(prefix = XxlJobProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@ConditionalOnClass(XxlJobSpringExecutor.class)
@EnableConfigurationProperties(XxlJobProperties.class)
public class CloudtXxlJobAutoConfiguration {

    @Value("${spring.application.name:unknown}")
    protected String applicationName;

    private final XxlJobProperties jobProperties;

    public CloudtXxlJobAutoConfiguration(XxlJobProperties jobProperties) {
        this.jobProperties = jobProperties;
    }

    @Bean
    @ConditionalOnMissingBean
    public XxlJobSpringExecutor xxlJobExecutor() {
        log.info(">>>>>>>>>>> xxl-job config init.");
        Assert.state(StringUtils.hasText(jobProperties.getAdminAddress()), "xxl-job的admin-address为空");

        XxlJobSpringExecutor xxlJobSpringExecutor = new XxlJobSpringExecutor();
        xxlJobSpringExecutor.setAdminAddresses(jobProperties.getAdminAddress());
        xxlJobSpringExecutor.setAppname(StrUtil.blankToDefault(jobProperties.getAppname(), applicationName));

        xxlJobSpringExecutor.setAddress(jobProperties.getAddress());
        xxlJobSpringExecutor.setIp(jobProperties.getIp());
        xxlJobSpringExecutor.setPort(jobProperties.getPort());
        xxlJobSpringExecutor.setAccessToken(jobProperties.getAccessToken());
        xxlJobSpringExecutor.setLogPath(jobProperties.getLogPath());
        xxlJobSpringExecutor.setLogRetentionDays(jobProperties.getLogRetentionDays());
        return xxlJobSpringExecutor;
    }

    /**
     * 针对多网卡、容器内部署等情况，可借助 "spring-cloud-commons" 提供的 "InetUtils" 组件灵活定制注册IP；
     *
     *      1、引入依赖：
     *          <dependency>
     *             <groupId>org.springframework.cloud</groupId>
     *             <artifactId>spring-cloud-commons</artifactId>
     *             <version>${version}</version>
     *         </dependency>
     *
     *      2、配置文件，或者容器启动变量
     *          spring.cloud.inetutils.preferred-networks: 'xxx.xxx.xxx.'
     *
     *      3、获取IP
     *          String ip_ = inetUtils.findFirstNonLoopbackHostInfo().getIpAddress();
     */

    @Bean
    public JobTrace xxlJobTrace() {
        return new JobTrace(applicationName);
    }

    @Bean
    public TenantJobAspect tenantJobAspect(TenantClientProvider tenantClientProvider,
                                           TenantDataIsolateProvider tenantDataIsolateProvider) {
        return new TenantJobAspect(tenantClientProvider, tenantDataIsolateProvider);
    }
}
