package com.elitescloud.boot.xxljob.config.tenant;

import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.xxl.job.core.CloudtJobInterceptor;
import com.xxl.job.core.biz.model.ReturnT;
import lombok.extern.log4j.Log4j2;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;

import javax.annotation.Nullable;
import javax.validation.constraints.NotBlank;
import java.util.Arrays;

/**
 * 租户上下文.
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/6
 */
@Log4j2
@Order(Ordered.HIGHEST_PRECEDENCE + 1)
public class XxlJobTenantInterceptor implements CloudtJobInterceptor {
    private static final String PARAM = "tenantCode";

    private final TenantClientProvider tenantClientProvider;

    public XxlJobTenantInterceptor(TenantClientProvider tenantClientProvider) {
        this.tenantClientProvider = tenantClientProvider;
    }

    @Override
    public boolean preExecute(@NotBlank String executorHandler, @Nullable String executorParams, int executorTimeout) {
        if (StrUtil.isBlank(executorParams)) {
            log.info("xxl-job执行参数：{}", executorParams);
            return true;
        }
        if (!tenantClientProvider.enabledTenant()) {
            // 已禁用租户
            log.info("租户已禁用");
            return true;
        }

        // 解析租户编码
        String tenantCode = Arrays.stream(executorParams.split("&"))
                .map(t -> t.split("="))
                .filter(t -> PARAM.equalsIgnoreCase(t[0]))
                .map(t -> t[1])
                .filter(StringUtils::hasText)
                .findFirst()
                .orElse(null);
        log.info("xxl-job执行参数：{}，{}", executorParams, tenantCode);
        if (StrUtil.isBlank(tenantCode)) {
            return true;
        }

        // 获取租户
        var tenant = tenantClientProvider.getTenantByCode(tenantCode);
        if (tenant == null) {
            log.error("租户{}不存在", tenantCode);
            return true;
        }

        // 设置上下文
        TenantContextHolder.setCurrentTenant(tenant);

        return true;
    }

    @Override
    public void afterCompletion(@NotBlank String executorHandler, @Nullable String executorParams, ReturnT<String> result, @Nullable Throwable throwable) {
        TenantContextHolder.clearCurrentTenant();
    }
}
