package com.elitescloud.cloudt.context.util;

import cn.hutool.core.collection.CollUtil;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * 集合工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/05
 */
public class CollectionUtil {

    private CollectionUtil() {
    }

    /**
     * 拆分集合
     *
     * @param collection 原集合
     * @param size       每次拆分的大小
     * @param consumer   接收子集合的consumer
     * @param <E>        集合中元素类型
     */
    public static <E> void split(@NotEmpty List<E> collection, int size, @NotNull Consumer<List<E>> consumer) {
        if (collection == null || collection.isEmpty()) {
            return;
        }
        size = Integer.max(1, size);

        int max = collection.size();
        if (size >= max) {
            consumer.accept(collection);
            return;
        }

        int index = 0;
        int tempMax = 0;
        List<E> sub = null;
        while (true) {
            tempMax = Integer.min(index + size, max);

            sub = new ArrayList<>(size);
            for (; index < tempMax; index++) {
                sub.add(collection.get(index));
            }

            consumer.accept(sub);
            if (tempMax == max) {
                break;
            }
        }
    }

    /**
     * 展开树
     * <p>
     * 树形数据转list
     *
     * @param collection  数据集合
     * @param getChildren 获取子列表的方法
     * @param <E>         数据类型
     * @return 展开后的列表
     */
    public static <E> List<E> expandTree(@NotEmpty List<E> collection, @NotNull Function<E, List<E>> getChildren) {
        if (CollectionUtils.isEmpty(collection)) {
            return Collections.emptyList();
        }

        List<E> resultList = new ArrayList<>(128);
        expand(collection, getChildren, resultList);
        return resultList;
    }

    /**
     * 截取集合
     *
     * @param collection 集合
     * @param offset     开始位置
     * @param size       数量
     * @param <E>        数据类型
     * @return 集合
     */
    public static <E> List<E> sub(@NotEmpty Collection<E> collection, int offset, int size) {
        int end = offset + size;
        return CollUtil.sub(collection, offset, end);
    }

    private static <E> void expand(@NotNull List<E> collection, @NotNull Function<E, List<E>> getChildren, List<E> resultList) {
        if (CollectionUtils.isEmpty(collection)) {
            return;
        }

        for (E e : collection) {
            resultList.add(e);

            expand(getChildren.apply(e), getChildren, resultList);
        }
        collection.clear();
    }
}
