package com.elitescloud.cloudt.context.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.TimeZone;

/**
 * 时间日期工具.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/24
 */
public class DatetimeUtil {

    public static final String FORMATTER_DATETIME_STR = "yyyy-MM-dd HH:mm:ss";
    /**
     * 常用日期时间格式
     */
    public static final DateTimeFormatter FORMATTER_DATETIME = DateTimeFormatter.ofPattern(FORMATTER_DATETIME_STR);
    public static final String FORMATTER_DATE_STR = "yyyy-MM-dd";
    public static final DateTimeFormatter FORMATTER_DATE = DateTimeFormatter.ofPattern(FORMATTER_DATE_STR);
    public static final String FORMATTER_TIME_STR = "HH:mm:ss";
    public static final DateTimeFormatter FORMATTER_TIME = DateTimeFormatter.ofPattern(FORMATTER_TIME_STR);
    public static final String FORMATTER_DATETIME_LONG_STR = "yyyyMMddHHmmssSSS";
    public static final DateTimeFormatter FORMATTER_DATETIME_LONG = DateTimeFormatter.ofPattern(FORMATTER_DATETIME_LONG_STR);
    public static final SimpleDateFormat FORMAT_DATE = new SimpleDateFormat(FORMATTER_DATE_STR);
    public static final SimpleDateFormat FORMAT_DATETIME = new SimpleDateFormat(FORMATTER_DATETIME_STR);

    /**
     * 服务器时区
     */
    public static final ZoneOffset SYS_ZONE_OFFSET = ZoneOffset.of("+8");
    public static final TimeZone SYS_TIME_ZONE = TimeZone.getTimeZone(SYS_ZONE_OFFSET);

    /**
     * 当前时间字符串
     *
     * @return 当前时间字符串
     */
    public static String currentTimeStr() {
        return FORMATTER_DATETIME.format(LocalDateTime.now());
    }

    /**
     * 当前时间字符串
     *
     * @return 当前时间字符串
     */
    public static String currentTimeLong() {
        return FORMATTER_DATETIME_LONG.format(LocalDateTime.now());
    }

    /**
     * date转LocalDateTime
     *
     * @param date date
     * @return LocalDateTime
     */
    public static LocalDateTime date2LocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant().atOffset(SYS_ZONE_OFFSET).toLocalDateTime();
    }

    /**
     * localDateTime转date
     *
     * @param dateTime localDateTime
     * @return date
     */
    public static Date localDateTime2Date(LocalDateTime dateTime) {
        if (dateTime == null) {
            return null;
        }
        return Date.from(dateTime.atZone(SYS_ZONE_OFFSET).toInstant());
    }

    /**
     * instant转LocalDateTime
     *
     * @param instant instant
     * @return LocalDateTime
     */
    public static LocalDateTime toLocalDateTime(Instant instant) {
        if (instant == null) {
            return null;
        }
        return instant.atZone(SYS_ZONE_OFFSET).toLocalDateTime();
    }

    /**
     * LocalDateTime转Instant
     *
     * @param localDateTime LocalDateTime
     * @return instant
     */
    public static Instant toInstant(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.toInstant(SYS_ZONE_OFFSET);
    }

    /**
     * 日期字符串转date
     *
     * @param date 日期格式
     * @return date
     */
    public static Date parseForDate(String date) {
        try {
            return FORMAT_DATE.parse(date);
        } catch (ParseException e) {
            throw new IllegalArgumentException("日期格式不正确");
        }
    }

    /**
     * 日期字符串转date
     *
     * @param date 日期格式
     * @return date
     */
    public static Date parseForDatetime(String date) {
        try {
            return FORMAT_DATETIME.parse(date);
        } catch (ParseException e) {
            throw new IllegalArgumentException("日期格式不正确");
        }
    }
}
