package com.elitescloud.cloudt.context.util;

import org.hibernate.validator.HibernateValidator;
import org.springframework.util.StringUtils;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * 校验工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/13
 */
public class ValidateUtil {

    private ValidateUtil() {
    }

    private static final Validator VALIDATOR_FAST = Validation.byProvider(HibernateValidator.class).configure().failFast(true).buildValidatorFactory().getValidator();
    private static final Validator VALIDATOR_ALL = Validation.byProvider(HibernateValidator.class).configure().failFast(false).buildValidatorFactory().getValidator();

    public static final Pattern PATTERN_IP = Pattern.compile("((2(5[0-5]|[0-4]\\d))|[0-1]?\\d{1,2})(\\.((2(5[0-5]|[0-4]\\d))|[0-1]?\\d{1,2})){3}");
    public static final Pattern PATTERN_MOBILE = Pattern.compile("(?:0|86|\\+86)?1[3-9]\\d{9}");
    public static final Pattern PATTERN_EMAIL = Pattern.compile("(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|\"(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21\\x23-\\x5b\\x5d-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])*\")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21-\\x5a\\x53-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])+)])");
    public static final Pattern PATTERN_PHONE = Pattern.compile("(010|02\\d|0[3-9]\\d{2})-?(\\d{6,8})");

    /**
     * 校验对象
     *
     * @param pojo 待检查对象
     * @param fast 如果为true，检查到第一个不合法的字段后就停止，后续字段不再校验
     * @param <T>  对象类型
     * @return 检查结果
     */
    public static <T> Set<ConstraintViolation<T>> validateObject(T pojo, boolean fast) {
        return VALIDATOR_FAST.validate(pojo);
    }

    /**
     * 是否是手机号
     *
     * @param mobile 手机号
     * @return 是否是手机号
     */
    public static boolean isMobile(String mobile) {
        return StringUtils.hasText(mobile) && PATTERN_MOBILE.matcher(mobile).matches();
    }

    /**
     * 是否是邮箱
     *
     * @param email 邮箱
     * @return 是否是邮箱
     */
    public static boolean isEmail(String email) {
        return StringUtils.hasText(email) && PATTERN_EMAIL.matcher(email).matches();
    }

    /**
     * 是否是座机号码
     *
     * @param phone 座机号码
     * @return 是否是座机号码
     */
    public static boolean isPhone(String phone) {
        return StringUtils.hasText(phone) && PATTERN_PHONE.matcher(phone).matches();
    }

    /**
     * 判断是否是IP
     *
     * @param ip 是否是IP
     * @return 是否是IP
     */
    public static boolean isIp(String ip) {
        return StringUtils.hasText(ip) && PATTERN_IP.matcher(ip).matches();
    }
}
