package com.elitescloud.boot.context;

import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import org.springframework.util.Assert;

/**
 * 租户数据session.
 * <p>
 * 当上下文中的租户不符合时，可使用该类切换数据所属租户，session中的租户优先级高于上下文中的租户
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/6
 */
public class TenantSession {

    /**
     * 租户上下文
     */
    private static final ThreadLocal<SysTenantDTO> CURRENT_TENANT = new ThreadLocal<>();
    /**
     * 是否使用默认租户
     */
    private static final ThreadLocal<Boolean> USE_DEFAULT = new ThreadLocal<>();
    /**
     * 无租户，不隔离
     */
    private static final ThreadLocal<Boolean> NO_TENANT = new ThreadLocal<>();

    protected TenantSession() {
    }

    /**
     * 获取当前租户信息
     *
     * @return 租户信息
     */
    public static SysTenantDTO getCurrentTenant() {
        return CURRENT_TENANT.get();
    }

    /**
     * 设置当前租户信息
     *
     * @param tenant 租户信息
     */
    public static void setCurrentTenant(SysTenantDTO tenant) {
        Assert.notNull(tenant, "当前租户信息为空");
        CURRENT_TENANT.set(tenant);
    }

    /**
     * 清除当前租户信息
     */
    public static void clearCurrentTenant() {
        CURRENT_TENANT.remove();
    }

    /**
     * 获取是否使用默认租户
     *
     * @return 是否使用默认租户
     */
    public static boolean getUseDefault() {
        var val = USE_DEFAULT.get();
        return val != null && val;
    }

    /**
     * 设置使用默认租户
     */
    public static void setUseDefault() {
        USE_DEFAULT.set(true);
    }

    /**
     * 清除使用默认租户
     */
    public static void clearUseDefault() {
        USE_DEFAULT.remove();
    }

    /**
     * 设置不使用租户
     */
    public static void setNoTenant() {
        NO_TENANT.set(true);
    }

    /**
     * 判断是否不使用租户
     *
     * @return 是否不使用租户
     */
    public static boolean getNoTenant() {
        var val = NO_TENANT.get();
        return val != null && val;
    }

    /**
     * 清除使用不使用租户
     */
    public static void clearNoTenant() {
        NO_TENANT.remove();
    }
}
