package com.elitescloud.boot.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

/**
 * JSON工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/6/1
 */
public class JSONUtil {
    private static final Logger LOG = LoggerFactory.getLogger(JSONUtil.class);
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.instance();

    /**
     * 对象转JSON字符串
     *
     * @param obj 对象
     * @return 字符串
     */
    public static String toJsonString(Object obj) {
        return toJsonString(obj, false);
    }

    /**
     * 对象转JSON字符串
     *
     * @param obj    对象
     * @param safely 安全转换
     * @return 字符串
     */
    public static String toJsonString(Object obj, boolean safely) {
        if (obj == null) {
            return null;
        }

        if (obj.getClass().isPrimitive()) {
            return obj.toString();
        }

        try {
            return OBJECT_MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            if (safely) {
                throw new IllegalArgumentException("转换JSON字符串异常", e);
            }
            LOG.error("转换JSON字符串异常：", e);
            return null;
        }
    }

    /**
     * json字符串转对象
     *
     * @param json  json字符串
     * @param clazz 对象类型class
     * @param <T>   对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, Class<T> clazz) {
        return json2Obj(json, clazz, false);
    }

    /**
     * json字符串转对象
     *
     * @param json  json字符串
     * @param clazz 对象类型class
     * @param <T>   对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, Class<T> clazz, boolean safely) {
        if (!StringUtils.hasText(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, clazz);
        } catch (JsonProcessingException e) {
            if (safely) {
                throw new IllegalArgumentException("JSON字符串转对象异常：", e);
            }
            LOG.error("json字符串【{}】转对象异常:{}", json, e.getMessage());
            return null;
        }
    }

    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json json字符串
     * @param <T>  对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json) {
        return json2Obj(json, false);
    }

    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json   json字符串
     * @param safely 安全转换
     * @param <T>    对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, boolean safely) {
        if (!StringUtils.hasText(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, new TypeReference<>() {
            });
        } catch (JsonProcessingException e) {
            if (safely) {
                throw new IllegalArgumentException("JSON字符串转对象异常：", e);
            }
            LOG.error("json字符串【{}】转对象异常:{}", json, e.getMessage());
            return null;
        }
    }
}
