package com.elitescloud.boot;

import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.Map;

/**
 * Spring Context工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/18
 */
@Log4j2
public class SpringContextHolder {

    private static ApplicationContext applicationContext;

    protected SpringContextHolder() {
    }

    /**
     * 判断是否已初始化
     *
     * @return 是否已初始化
     */
    public static boolean initialized() {
        return applicationContext != null;
    }

    /**
     * 从spring context获取bean
     *
     * @param clazz bean class
     * @param <T>   bean类型
     * @return bean
     */
    public static <T> T getBean(Class<T> clazz) {
        assertContextInjected();
        Assert.notNull(clazz, "bean类型为空");

        return applicationContext.getBean(clazz);
    }

    /**
     * 从spring context获取bean
     *
     * @param name bean名称
     * @return bean
     */
    public static Object getBean(String name) {
        assertContextInjected();
        Assert.hasText(name, "bean名称为空");

        return applicationContext.getBean(name);
    }

    /**
     * 从spring context获取bean
     *
     * @param name  bean名称
     * @param clazz bean class
     * @param <T>   bean类型
     * @return bean
     */
    public static <T> T getBean(String name, Class<T> clazz) {
        assertContextInjected();
        Assert.hasText(name, "bean名称为空");
        Assert.notNull(clazz, "bean类型为空");

        return applicationContext.getBean(name, clazz);
    }

    /**
     * 从spring context获取bean
     *
     * @param clazz bean class
     * @param <T>   bean类型
     * @return bean名称与bean
     */
    public static <T> Map<String, T> getBeans(Class<T> clazz) {
        assertContextInjected();
        Assert.notNull(clazz, "bean类型为空");

        return applicationContext.getBeansOfType(clazz);
    }

    /**
     * 获取beanProvider
     *
     * @param clazz bean
     * @param <T>   bean类型
     * @return beanProvider
     */
    public static <T> ObjectProvider<T> getObjectProvider(Class<T> clazz) {
        assertContextInjected();
        Assert.notNull(clazz, "bean类型为空");

        return applicationContext.getBeanProvider(clazz);
    }

    /**
     * 获取配置
     *
     * @param key key
     * @return 值
     */
    public static String getProperty(String key) {
        return getProperty(key, String.class, null);
    }

    /**
     * 获取配置
     *
     * @param key          key
     * @param defaultValue 默认值
     * @return 值
     */
    public static String getProperty(String key, String defaultValue) {
        return getProperty(key, String.class, defaultValue);
    }

    /**
     * 获取配置
     *
     * @param key       key
     * @param valueType value类型
     * @param <T>       值类型
     * @return 值
     */
    public static <T> T getProperty(String key, Class<T> valueType) {
        return getProperty(key, valueType, null);
    }

    /**
     * 获取配置
     *
     * @param key          key
     * @param valueType    value类型
     * @param defaultValue 默认值
     * @param <T>          值类型
     * @return 值
     */
    public static <T> T getProperty(String key, Class<T> valueType, T defaultValue) {
        Assert.hasText(key, "key为空");
        Assert.notNull(valueType, "值类型为空");

        Environment environment = getBean(Environment.class);
        T value = environment.getProperty(key, valueType);
        if (defaultValue == null) {
            return value;
        }

        if (value == null) {
            return defaultValue;
        }
        if (value instanceof String && !StringUtils.hasText((String) value)) {
            return defaultValue;
        }
        return value;
    }

    /**
     * 获取指定的资源
     *
     * @param location 资源路径
     * @return 资源
     */
    public static Resource getResource(String location) {
        Assert.hasText(location, "location为空");
        assertContextInjected();
        return applicationContext.getResource(location);
    }

    /**
     * 获取指定的资源
     *
     * @param locationPattern 资源路径格式
     * @return 资源
     */
    public static Resource[] getResources(String locationPattern) {
        Assert.hasText(locationPattern, "locationPattern为空");
        assertContextInjected();
        try {
            return applicationContext.getResources(locationPattern);
        } catch (IOException e) {
            throw new BusinessException("获取资源文件【" + locationPattern + "】失败", e);
        }
    }

    /**
     * 获取应用名称
     *
     * @return 应用名称
     */
    public static String getApplicationName() {
        return getProperty("spring.application.name");
    }

    /**
     * 装配bean的依赖
     * <p>
     * 支持spring的相关依赖注入，包含Autowired、Resource、Inject等注解
     *
     * @param bean bean
     */
    public static void autowireBean(Object bean) {
        Assert.notNull(bean, "对象为空");
        assertContextInjected();

        applicationContext.getAutowireCapableBeanFactory().autowireBean(bean);
    }

    static void setApplicationContext(ApplicationContext applicationContext) {
        SpringContextHolder.applicationContext = applicationContext;
    }

    private static void assertContextInjected() {
        if (applicationContext == null) {
            throw new IllegalStateException("请先设置applicationContext.");
        }
    }
}
