package com.elitescloud.boot;

import com.elitescloud.boot.support.app.CloudtAppConfig;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.log4j.Log4j2;
import org.springframework.aop.interceptor.AsyncUncaughtExceptionHandler;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.SimpleApplicationEventMulticaster;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.task.TaskExecutor;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.support.TaskUtils;

import java.util.concurrent.Executor;

/**
 * Cloudt Context主配置类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/18
 */
@Import({CloudtContextAutoConfiguration.JacksonConfiguration.class, CloudtContextAutoConfiguration.AsyncConfiguration.class, CloudtAppConfig.class})
@EnableConfigurationProperties({CloudtContextProperties.class})
@Order(Ordered.HIGHEST_PRECEDENCE)
@Log4j2
class CloudtContextAutoConfiguration implements ApplicationContextAware, InitializingBean {

    @Autowired
    private SimpleApplicationEventMulticaster applicationEventMulticaster;

    @Override
    public void setApplicationContext(@NonNull ApplicationContext applicationContext) throws BeansException {
        onApplicationContext(applicationContext);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        applicationEventMulticaster.setErrorHandler(TaskUtils.LOG_AND_SUPPRESS_ERROR_HANDLER);
    }

    private void onApplicationContext(ApplicationContext applicationContext) {
        SpringContextHolder.setApplicationContext(applicationContext);
    }

    static class JacksonConfiguration {

        @Bean
        public Jackson2ObjectMapperBuilderCustomizer jackson2ObjectMapperBuilderCustomizer() {
            return ObjectMapperFactory::builderInstance;
        }

        @Bean
        public LocalDateTimeSerializer localDateTimeSerializer() {
            return new LocalDateTimeSerializer(DatetimeUtil.FORMATTER_DATETIME);
        }

        @Bean
        public LocalDateTimeDeserializer localDateTimeDeserializer() {
            return new LocalDateTimeDeserializer(DatetimeUtil.FORMATTER_DATETIME);
        }
    }

    @EnableAsync
    @Log4j2
    static class AsyncConfiguration implements AsyncConfigurer {

        private final TaskExecutor taskExecutor;

        public AsyncConfiguration(TaskExecutor taskExecutor) {
            this.taskExecutor = taskExecutor;
        }

        @Override
        public Executor getAsyncExecutor() {
            return taskExecutor;
        }

        @Override
        public AsyncUncaughtExceptionHandler getAsyncUncaughtExceptionHandler() {
            return (ex, method, params) -> log.error("异步线程【" + method.getDeclaringClass().getName() + "." + method.getName() + "】执行异常：", ex);
        }
    }
}
