package com.elitescloud.cloudt.security.entity;

import com.elitescloud.cloudt.security.CurrentUserExtension;
import com.elitescloud.cloudt.security.common.InnerUserEnum;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;
import lombok.experimental.Accessors;
import org.springframework.security.core.CredentialsContainer;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Michael Li
 * AC 2021-04-11:16:12
 */
@Data
@Accessors(chain = true)
public class GeneralUserDetails implements UserDetails, CredentialsContainer {

    private static final long serialVersionUID = 414652772107486653L;

    /**
     * 权限（角色）
     */
    private List<GrantedAuthority> authorities;
    /**
     * 用户详细信息
     */
    private SysUserDTO user;
    /**
     * 登录IP
     */
    private String ip;
    /**
     * 浏览器
     */
    private String browser;
    /**
     * 登录地址
     */
    private String address;

    /**
     * 增加返回当前用户所属的BU IDs，包括用户当前所属BU，以及这个BU
     */
    @Deprecated
    private List<Long> selfBuIds;

    /**
     * 除用户所属的BU IDs，还有离散授权的BU IDs
     */
    @Deprecated
    private List<Long> grantedBuIds;

    /**
     * 支持业务自定义的扩展信息
     * {@link CurrentUserExtension}
     */
    private Object extendInfo;

    public GeneralUserDetails() {
    }

    public GeneralUserDetails(SysUserDTO sysUserDTO) {
        this.user = sysUserDTO;
    }

    public List<String> getRoleCodes() {
        if (user.getRoleCodes() == null) {
            return Collections.emptyList();
        }
        return new ArrayList<>(user.getRoleCodes());
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        if (user.getRoleCodes() == null) {
            return Collections.emptyList();
        }
        return user.getRoleCodes().stream()
                .map(SimpleGrantedAuthority::new).collect(Collectors.toList());
    }

    @JsonIgnore
    @Override
    public String getPassword() {
        return user.getPassword();
    }

    @Override
    public String getUsername() {
        return user.getUsername();
    }

    @Override
    public boolean isAccountNonExpired() {
        return user.getExpiredTime() == null || user.getExpiredTime().isAfter(LocalDateTime.now());
    }

    @Override
    public boolean isAccountNonLocked() {
        return !Boolean.TRUE.equals(user.getLocked());
    }

    @Override
    public boolean isCredentialsNonExpired() {
        if (user.getPwdExpiredTime() == null) {
            return true;
        }

        return LocalDateTime.now().isBefore(user.getPwdExpiredTime());
    }

    @Override
    public boolean isEnabled() {
        return user.getEnabled() != null && user.getEnabled();
    }

    @Override
    public void eraseCredentials() {
        user.setPassword(null);
    }

    public Long getUserId() {
        return getUser().getId();
    }

    public Long getOrgId() {
        if (user.getOrg() == null) {
            return null;
        }
        return user.getOrg().getId();
    }

    public Long getTenantOrgId() {
        if (user.getTenantOrg() == null) {
            return null;
        }
        return user.getTenantOrg().getId();
    }

    /**
     * 是否是系统管理员
     *
     * @return 是否是系统管理员
     */
    public boolean isSystemAdmin() {
        return InnerUserEnum.ADMIN.getUsername().equals(getUsername());
    }

    /**
     * 是否是租户管理员
     *
     * @return 是否是租户管理员
     */
    public boolean isTenantAdmin() {
        return getTenant() != null && Objects.equals(getUserId(), getTenant().getSysUserId());
    }

    /**
     * 是否是租户组织管理员
     *
     * @return 是否是租户组织管理员
     */
    public boolean isTenantOrgAdmin() {
        if (user.getTenantOrgAdminId() == null) {
            return false;
        }
        return user.getTenantOrgAdminId().longValue() == user.getId();
    }

    /**
     * 是否是普通的租户下的用户（除了管理员）
     *
     * @return 是否是普通的租户下的用户
     */
    public boolean isTenantUser() {
        return getTenant() != null && !Objects.equals(getUserId(), getTenant().getSysUserId()) && getTenant().getType() != TenantType.OPERATION;
    }

    /**
     * 是否属于运营机构
     *
     * @return 是否属于运营机构
     */
    public boolean isOperation() {
        if (isSystemAdmin()) {
            return true;
        }
        return getTenant() != null && getTenant().getType() == TenantType.OPERATION;
    }

    /**
     * 是否是普通的用户
     * <p>
     * 并非租户下的用户
     *
     * @return 是否是普通的用户
     */
    public boolean isCommonUser() {
        return getTenant() == null && !InnerUserEnum.ADMIN.getUsername().equals(getUsername());
    }

    public SysTenantDTO getTenant() {
        return user.getSysTenantVO();
    }

    public Long getTenantId() {
        if (user.getSysTenantVO() == null) {
            return null;
        }
        return user.getSysTenantVO().getId();
    }
}
