package com.elitescloud.boot.condition.support;

import com.elitescloud.boot.condition.ConditionalOnRpc;
import org.springframework.boot.autoconfigure.condition.ConditionOutcome;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.util.StringUtils;

import java.util.Collections;

/**
 * 启用RPC的条件.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/28
 */
public class OnRpcCondition extends AbstractCondition {
    private static final String TRUE = "true";

    @Override
    public ConditionOutcome getMatchOutcome(ConditionContext context, AnnotatedTypeMetadata metadata) {
        // 获取注解的属性
        var attributes = metadata.getAnnotationAttributes(ConditionalOnRpc.class.getName());
        if (attributes == null) {
            attributes = Collections.emptyMap();
        }

        // 是否必须dubbo
        boolean enabledDubbo = enabledDubbo(context);
        if (!enabledDubbo && existsAttribute(attributes, "requiredDubbo", TRUE)) {
            return ConditionOutcome.noMatch("未启用dubbo");
        }

        // 是否必须OpenFeign
        boolean enabledOpenFeign = enabledOpenFeign(context);
        if (!enabledOpenFeign && existsAttribute(attributes, "requiredOpenFeign", TRUE)) {
            return ConditionOutcome.noMatch("未启用OpenFeign");
        }

        if (enabledDubbo || enabledOpenFeign) {
            return ConditionOutcome.match();
        }
        return ConditionOutcome.noMatch("未启用RPC");
    }

    /**
     * 是否启用dubbo
     *
     * @param context
     * @return
     */
    private boolean enabledDubbo(ConditionContext context) {
        boolean present = isPresent("org.apache.dubbo.config.bootstrap.DubboBootstrap", null);
        if (!present) {
            return false;
        }

        String enabled = context.getEnvironment().getProperty("dubbo.enabled");
        return !StringUtils.hasText(enabled) || enabled.equalsIgnoreCase(TRUE);
    }

    /**
     * 是否启用openFeign
     *
     * @param context
     * @return
     */
    private boolean enabledOpenFeign(ConditionContext context) {
        return isPresent("org.springframework.cloud.openfeign.FeignClient", null);
    }
}
