package com.elitescloud.cloudt.context.util;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.extra.servlet.ServletUtil;
import cn.hutool.http.useragent.Browser;
import cn.hutool.http.useragent.UserAgentUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.context.CloudtRequestContextHolder;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.server.PathContainer;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.util.pattern.PathPatternParser;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 * HttpServletUtil.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/13
 */
@Log4j2
public class HttpServletUtil {
    private static final Map<Class<?>, Object> SHARED_OBJECTS = new HashMap<>();
    private static final AntPathMatcher ANT_PATH_MATCHER = new AntPathMatcher();
    public static final String PROTOCOL_HTTPS = "https://";
    public static final String PROTOCOL_HTTP = "http://";

    static {
        Browser.addCustomBrowser("Postman", "PostmanRuntime", "PostmanRuntime" + Browser.Other_Version);
    }

    private HttpServletUtil() {
    }

    /**
     * 获取客户端真实IP
     *
     * @return 客户端IP
     */
    public static String currentClientIp() {
        HttpServletRequest request = currentRequest();
        if (request == null) {
            return null;
        }
        return ServletUtil.getClientIP(request);
    }

    /**
     * 获取当前请求
     *
     * @return request
     */
    public static HttpServletRequest currentRequest() {
        RequestAttributes requestAttributes = CloudtRequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            requestAttributes = RequestContextHolder.getRequestAttributes();
        }
        if (requestAttributes == null) {
            return null;
        }

        return ((ServletRequestAttributes) requestAttributes).getRequest();
    }

    /**
     * 解析域名
     *
     * @param request request
     * @return 域名
     */
    public static String obtainDomain(@NotNull HttpServletRequest request) {
        String ip = obtainDomain(request.getRequestURL().toString());
        if (!com.elitescloud.boot.util.ValidateUtil.isIp(ip)) {
            return ip;
        }

        // 从请求头获取
        String domain = request.getHeader(HttpHeaders.HOST);
        if (StringUtils.hasText(domain) && !com.elitescloud.boot.util.ValidateUtil.isIp(domain.split(":")[0])) {
            return domain;
        }

        // 从请求头获取
        domain = request.getHeader(HttpHeaders.REFERER);
        if (StringUtils.hasText(domain)) {
            domain = obtainDomain(domain);
            if (!com.elitescloud.boot.util.ValidateUtil.isIp(domain)) {
                return domain;
            }
        }

        // 从请求头获取
        domain = request.getHeader(HttpHeaders.ORIGIN);
        if (StringUtils.hasText(domain)) {
            domain = obtainDomain(domain);
            if (!com.elitescloud.boot.util.ValidateUtil.isIp(domain)) {
                return domain;
            }
        }

        return ip;
    }

    /**
     * 解析域名
     *
     * @param url request路径
     * @return 域名
     */
    public static String obtainDomain(@NotBlank String url) {
        boolean isHttpProtocol = false;
        if (url.startsWith(PROTOCOL_HTTP)) {
            url = url.substring(7);
            isHttpProtocol = true;
        } else if (url.startsWith(PROTOCOL_HTTPS)) {
            url = url.substring(8);
            isHttpProtocol = true;
        }

        if (!isHttpProtocol) {
            return null;
        }

        int lastIndex = url.indexOf("/");
        if (lastIndex > 0) {
            url = url.substring(0, lastIndex);
        }
        lastIndex = url.indexOf(":");
        if (lastIndex > 0) {
            url = url.substring(0, lastIndex);
        }

        return url;
    }

    /**
     * 解析请求参数
     * <p>
     * 包含查询参数和表单参数
     *
     * @param request
     * @return
     */
    public static MultiValueMap<String, String> getParameters(@NotNull HttpServletRequest request) {
        Map<String, String[]> parameterMap = request.getParameterMap();
        MultiValueMap<String, String> parameters = new LinkedMultiValueMap<>(parameterMap.size());
        parameterMap.forEach((key, values) -> {
            if (values.length > 0) {
                for (String value : values) {
                    parameters.add(key, value);
                }
            } else {
                parameters.add(key, null);
            }
        });
        return parameters;
    }

    /**
     * 获取查询参数
     * <p>
     * 仅仅返回查询参数
     *
     * @param request
     * @return
     */
    public static MultiValueMap<String, String> getQueryParameters(@NotNull HttpServletRequest request) {
        String queryString = request.getQueryString();
        if (!StringUtils.hasText(queryString)) {
            return new LinkedMultiValueMap<>(0);
        }

        String[] parameterArray = queryString.split("&");
        MultiValueMap<String, String> parameters = new LinkedMultiValueMap<>(parameterArray.length);
        for (String s : parameterArray) {
            if (!StringUtils.hasText(s)) {
                continue;
            }
            String[] params = s.split("=");
            parameters.add(params[0], params.length == 2 ? params[1] : null);
        }

        return parameters;
    }

    /**
     * 获取表单参数
     * <p>
     * 仅仅返回表单部分参数
     *
     * @param request 仅仅返回
     * @return
     */
    public static MultiValueMap<String, String> getFormParameters(@NotNull HttpServletRequest request) {
        var queryParams = getQueryParameters(request);
        if (queryParams.isEmpty()) {
            return getParameters(request);
        }

        Map<String, String[]> parameterMap = request.getParameterMap();
        MultiValueMap<String, String> parameters = new LinkedMultiValueMap<>(parameterMap.size());
        parameterMap.forEach((key, values) -> {
            if (queryParams.containsKey(key)) {
                return;
            }
            if (values.length > 0) {
                for (String value : values) {
                    parameters.add(key, value);
                }
            } else {
                parameters.add(key, null);
            }
        });
        return parameters;
    }

    /**
     * 判断是否含有文件上传
     *
     * @param request
     * @return
     */
    public static boolean isMultipartContent(@NotNull ServletRequest request) {
        String contentType = request.getContentType();
        if (contentType == null) {
            return false;
        }

        return contentType.toLowerCase().startsWith("multipart/");
    }

    /**
     * 获取所有的请求头
     *
     * @param request request
     * @return 请求头
     */
    public static MultiValueMap<String, String> getHeaders(@NotNull HttpServletRequest request) {
        MultiValueMap<String, String> headerMap = new LinkedMultiValueMap<>(64);
        var names = request.getHeaderNames();
        while (names.hasMoreElements()) {
            var name = names.nextElement();
            var values = request.getHeaders(name);
            while (values.hasMoreElements()) {
                headerMap.add(name, values.nextElement());
            }
        }
        return headerMap;
    }

    /**
     * 获取浏览器类型
     *
     * @param request 请求
     * @return 浏览器类型
     */
    public static String getBrowser(@NotNull HttpServletRequest request) {
        var userAgent = request.getHeader(HttpHeaders.USER_AGENT);
        if (CharSequenceUtil.isBlank(userAgent)) {
            return null;
        }
        try {
            var agent = UserAgentUtil.parse(userAgent);
            return agent == null ? null : agent.getBrowser().getName();
        } catch (Exception e) {
            log.error("解析user-agent异常：", e);
        }
        return null;
    }

    /**
     * response返回json格式内容
     *
     * @param response response
     * @param object   待返回的对象
     * @throws Exception 异常
     */
    public static void writeJson(@NotNull HttpServletResponse response, @NotNull Object object) throws Exception {
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);

        var writer = response.getWriter();

        if (object instanceof String) {
            writer.write((String) object);
        } else {
            writer.write(getObjectMapper().writeValueAsString(object));
        }

        writer.flush();
        writer.close();
    }

    /**
     * response返回json格式内容
     * <p>
     * 忽略异常
     *
     * @param response response
     * @param object   待返回的对象
     */
    public static void writeJsonIgnoreException(@NotNull HttpServletResponse response, @NotNull Object object) {
        try {
            writeJson(response, object);
        } catch (Exception e) {
            log.info("返回前端信息失败：", e);
        }
    }

    /**
     * 是否匹配当前请求路径
     *
     * @param request 请求
     * @param pattern 匹配路径
     * @return 是否匹配
     */
    public static boolean isMatch(@NotNull HttpServletRequest request, @NotBlank String pattern) {
        String uri = request.getRequestURI();
        var pathPattern = PathPatternParser.defaultInstance.parse(pattern);
        return pathPattern.matches(PathContainer.parsePath(uri));
    }

    /**
     * 是否匹配当前请求路径
     *
     * @param request 请求
     * @param pattern 匹配路径
     * @return 是否匹配
     */
    public static boolean isMatchByAntPath(@NotNull HttpServletRequest request, @NotBlank String pattern) {
        String uri = request.getRequestURI();
        return ANT_PATH_MATCHER.match(pattern, uri);
    }

    private static ObjectMapper getObjectMapper() {
        var clazz = ObjectMapper.class;
        ObjectMapper objectMapper = getSharedObject(clazz);
        if (objectMapper != null) {
            return objectMapper;
        }

        objectMapper = SpringContextHolder.getBean(clazz);
        setSharedObject(clazz, objectMapper);
        return objectMapper;
    }

    private static <C> C getSharedObject(Class<C> clazz) {
        return (C) SHARED_OBJECTS.get(clazz);
    }

    private static <C> void setSharedObject(Class<C> clazz, C object) {
        SHARED_OBJECTS.put(clazz, object);
    }
}
