package com.elitescloud.boot;

import com.elitescloud.boot.context.ExecutorContextHolder;
import com.elitescloud.boot.context.properties.DataFormatterProperties;
import com.elitescloud.boot.provider.DataFormatterProvider;
import com.elitescloud.boot.provider.impl.DefaultDataFormatterProvider;
import com.elitescloud.boot.support.app.CloudtAppConfig;
import com.elitescloud.boot.support.serializer.BigDecimalSerializer;
import com.elitescloud.boot.support.transfer.ExcutorContextTransfer;
import com.elitescloud.boot.threadpool.common.ContextTransfer;
import com.elitescloud.boot.util.CloudtBootUtil;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.log4j.Log4j2;
import org.springframework.aop.interceptor.AsyncUncaughtExceptionHandler;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.info.Info;
import org.springframework.boot.actuate.info.InfoContributor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.SimpleApplicationEventMulticaster;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.task.TaskExecutor;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.support.TaskUtils;

import java.math.BigDecimal;
import java.util.concurrent.Executor;

/**
 * Cloudt Context主配置类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/18
 */
@Import({CloudtContextAutoConfiguration.JacksonConfiguration.class, CloudtContextAutoConfiguration.AsyncConfiguration.class, CloudtAppConfig.class})
@EnableConfigurationProperties({CloudtContextProperties.class, DataFormatterProperties.class})
@Order(Ordered.HIGHEST_PRECEDENCE)
@Log4j2
class CloudtContextAutoConfiguration implements InitializingBean {

    @Autowired
    private SimpleApplicationEventMulticaster applicationEventMulticaster;

    @Override
    public void afterPropertiesSet() throws Exception {
        applicationEventMulticaster.setErrorHandler(TaskUtils.LOG_AND_SUPPRESS_ERROR_HANDLER);
    }

    @Bean
    public CloudtActuatorInfo cloudtActuatorInfo() {
        return new CloudtActuatorInfo();
    }

    @Bean
    public ContextTransfer<ExecutorContextHolder.ExecutorContext> excutorContextTransfer() {
        return new ExcutorContextTransfer();
    }

    static class JacksonConfiguration {

        @Bean
        @ConditionalOnMissingBean
        public DataFormatterProvider dataFormatterProvider(DataFormatterProperties properties) {
            return new DefaultDataFormatterProvider(properties);
        }

        @Bean
        public Jackson2ObjectMapperBuilderCustomizer jackson2ObjectMapperBuilderCustomizerBigDecimal(
                DataFormatterProvider dataFormatterProvider) {
            return builder -> builder.serializerByType(BigDecimal.class, new BigDecimalSerializer(dataFormatterProvider));
        }

        @Bean
        public Jackson2ObjectMapperBuilderCustomizer jackson2ObjectMapperBuilderCustomizer() {
            return ObjectMapperFactory::builderInstance;
        }

        @Bean
        public LocalDateTimeSerializer localDateTimeSerializer() {
            return new LocalDateTimeSerializer(DatetimeUtil.FORMATTER_DATETIME);
        }

        @Bean
        public LocalDateTimeDeserializer localDateTimeDeserializer() {
            return new LocalDateTimeDeserializer(DatetimeUtil.FORMATTER_DATETIME);
        }
    }

    @EnableAsync
    @Log4j2
    static class AsyncConfiguration implements AsyncConfigurer {

        private final TaskExecutor taskExecutor;

        public AsyncConfiguration(TaskExecutor taskExecutor) {
            this.taskExecutor = taskExecutor;
        }

        @Override
        public Executor getAsyncExecutor() {
            return taskExecutor;
        }

        @Override
        public AsyncUncaughtExceptionHandler getAsyncUncaughtExceptionHandler() {
            return (ex, method, params) -> log.error("异步线程【" + method.getDeclaringClass().getName() + "." + method.getName() + "】执行异常：", ex);
        }
    }

    static class CloudtActuatorInfo implements InfoContributor {

        @Override
        public void contribute(Info.Builder builder) {
            builder.withDetail("Cloudt-Boot-Version", CloudtBootUtil.getVersion());
        }
    }
}
