package com.elitescloud.boot.context;

import lombok.Getter;
import org.springframework.util.Assert;

import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.time.LocalDateTime;

/**
 * 执行者上下文.
 *
 * @author Kaiser（wang shao）
 * @date 2024/2/28
 */
public class ExecutorContextHolder {
    private static final ThreadLocal<ExecutorContext> CURRENT_THREAD = new ThreadLocal<>();

    /**
     * 创建上下文
     *
     * @param source          来源
     * @param parent          父上下文
     * @param createIfPresent 如果已存在，是否还创建
     * @return 执行者上下文
     */
    public static ExecutorContext create(@NotNull Source source, ExecutorContext parent, boolean createIfPresent) {
        Assert.notNull(source, "来源不能为空");

        var executorContext = get();

        // 不存在则创建
        if (executorContext == null) {
            executorContext = new ExecutorContext(source, parent);
            CURRENT_THREAD.set(executorContext);
            return executorContext;
        }

        if (createIfPresent) {
            // 创建子上下文
            var executorContextChild = new ExecutorContext(source, executorContext);
            CURRENT_THREAD.set(executorContextChild);
            return executorContextChild;
        }

        return executorContext;
    }

    /**
     * 获取上下文
     *
     * @return 执行者上下文
     */
    public static ExecutorContext get() {
        return CURRENT_THREAD.get();
    }

    /**
     * 清除上下文
     */
    public static void clear() {
        CURRENT_THREAD.remove();
    }

    public static enum Source {
        /**
         * 来源
         */
        WEB("Web"),
        WEB_SOCKET("WebSocket"),
        WEB_SERVICE("WebService"),
        XXL_JOB("XXL-JOB任务调度"),
        MQ("MQ消息队列"),
        SYNC("异步线程");

        /**
         * 描述
         */
        private final String desc;

        Source(String desc) {
            this.desc = desc;
        }

        public String getDesc() {
            return desc;
        }
    }

    @Getter
    public static class ExecutorContext implements Serializable {
        private static final long serialVersionUID = -3606708282531782524L;

        /**
         * 创建时间
         */
        private final LocalDateTime createTime;

        private final Source source;
        /**
         * 父上下文
         */
        private final ExecutorContext parent;

        public ExecutorContext(Source source, ExecutorContext parent) {
            this.source = source;
            this.createTime = LocalDateTime.now();
            this.parent = parent;
        }

        @Override
        public String toString() {
            return "ExecutorContext{" +
                    "createTime=" + createTime +
                    ", source=" + source +
                    ", parent=" + parent +
                    '}';
        }
    }
}
